package com.jz.jar.media.proxy;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.common.utils.collection.OrderTool;
import com.jz.jar.media.common.ItemCard;
import com.jz.jar.media.enums.UploadVideoStatus;
import com.jz.jar.media.request.VideoRequest;
import com.jz.jar.media.service.UploadVideoService;
import com.jz.jar.media.tool.VideoTool;
import com.jz.jar.media.wrapper.VideoWrapper;
import com.jz.jooq.media.tables.pojos.UploadVideo;

/**
 * @Title UploadVideoProxy
 * @Package com.jz.jar.media.proxy
 * @author tangjunfeng
 * @date 2018年10月24日 上午11:49:03
 * @version V1.0
 */
@Lazy
@Service
public class UploadVideoProxy {
	
	/* 获取上传视频有效的状态 */
	private static final List<UploadVideoStatus> validStatus = Arrays.asList(UploadVideoStatus.online,
			UploadVideoStatus.wating, UploadVideoStatus.verifying, UploadVideoStatus.notPassed);
	
	@Autowired
	private UploadVideoService uploadVideoService;
	
	/** 是否学员上传的视频 */
	public boolean isStudentUploadVideo(String wid) {
		// oss.oss_identity_config.prefix
		return wid.startsWith("sv_");
	}
	
	/** 获取当前用户上传的视频总数 */
	public int countVideoForUid(String uid, String app) {
		return uploadVideoService.countVideos(uid, validStatus, app);
	}
	
	/** 获取当前用户上传的视频信息 */
	public List<ItemCard> getVideosForUid(String uid, String app, int start, int size) {
		List<String> wids = uploadVideoService.getVideos(uid, validStatus, app, start, size);
		if (ArrayMapTools.isEmpty(wids))
			return null;
		return packageVideoCard(VideoRequest.of(uid, wids));
	}
	
	public VideoWrapper packageVideoWrapper(UploadVideo video) {
		if (null == video)
			return null;
		VideoWrapper vw = VideoWrapper.of(video);
		// 只有已上线的视频才可以播放
		if (UploadVideoStatus.online.getCode() == video.getStatus()) {
			// 视频播放地址
			VideoTool.getVideoUrl(video, vw);
			// 图片
			VideoTool.getSnapshotUrl(video.getUid(), video, vw);
			return vw;
		}
		VideoTool.getUploadVideoDefault(vw);
		return vw;
	}
	
	public VideoWrapper packageVideoWrapper(String uid, UploadVideo video) {
		VideoWrapper vw = VideoWrapper.of(video);
		// 只有已上线的视频才可以播放
		if (UploadVideoStatus.online.getCode() == video.getStatus()) {
			// 视频播放地址
			VideoTool.getVideoUrl(video, vw);
			// 图片
			VideoTool.getSnapshotUrl(uid, video, vw);
			return vw;
		}
		VideoTool.getUploadVideoDefault(vw);
		return vw;
	}
	
	/** 将当前视频信息包装成卡片信息 */
	public List<ItemCard> packageVideoCard(VideoRequest request) {
		if (ArrayMapTools.isEmpty(request.getWids()))
			return null;
		List<UploadVideo> videos = uploadVideoService.getVideoInfo(request.getWids());
		if (ArrayMapTools.isEmpty(videos))
			return null;
		return packageVideoCard(request.getUid(), OrderTool.keepOrder(request.getWids(), videos, UploadVideo::getWid));
	}
	
	/** 将当前视频信息包装成卡片信息 */
	public List<ItemCard> packageVideoCard(String uid, List<UploadVideo> videos) {
		if (ArrayMapTools.isEmpty(videos))
			return null;
		return videos.stream().map(it -> ItemCard.of(packageVideoWrapper(uid, it))).collect(Collectors.toList());
	}
}
