package com.jz.jar.media.proxy;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import com.google.common.collect.Maps;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.jar.media.enums.DbField;
import com.jz.jar.media.request.NextPlayVideoRequest;
import com.jz.jar.media.service.CommonPlayHistoryService;
import com.jz.jar.media.service.PlaylistVideoService;
import com.jz.jar.media.service.UserPlayHistoryService;
import com.jz.jar.media.wrapper.VideoWrapper;
import com.jz.jooq.media.tables.pojos.CommonPlayHistory;
import com.jz.jooq.media.tables.pojos.Playlist;
import com.jz.jooq.media.tables.pojos.UserPlayHistory;

/**
 * @Title PlaylistVideoProxy
 * @Package com.jz.jar.media.proxy
 * @author tangjunfeng
 * @date 2018年8月8日 下午6:26:16
 * @version V1.0
 */
@Lazy
@Service
public class PlaylistVideoProxy {

	@Autowired
	private UserPlayHistoryService userPlayHistoryService;
	@Autowired
	private PlaylistVideoService playlistVideoService;
	@Autowired
	private CommonPlayHistoryService commonPlayHistoryService;
	@Autowired
	private VideoProxy videoProxy;

	public String getNextPlayVideo(String pid, String wid) {
		List<String> wids = playlistVideoService.findWids(pid);
		// 当前专辑是空的 || 当前专辑内无当前视频
		if (ArrayMapTools.isEmpty(wids) || !wids.contains(wid))
			return wid;
		// 当前视频为最后一集
		String lastWid = wids.get(wids.size() - 1);
		if (lastWid.equals(wid))
			return wids.get(0);
		// 读取当前视频的下一集
		for (int i = 0; i < wids.size(); i++) {
			if (wid.equals(wids.get(i))) {
				return wids.get(i + 1);
			}
		}
		return wid;
	}

	/** 获取专辑下一个需要播放的视频 */
	public VideoWrapper getNextPlayVideoForHistory(String uid, Playlist playlist) {
		int playlength = 0;
		String nextWid = playlist.getWid();
		if (null != uid) {
			// 获取当前专辑的播放历史
			UserPlayHistory history = userPlayHistoryService.getHistory(uid, playlist.getPid());
			if (null != history) {
				playlength = history.getPlayLength();
				nextWid = history.getNextWid();
			}
		}
		VideoWrapper videoInfo = videoProxy.getVideoInfo(uid, playlist.getPid(), nextWid);
		if (null != videoInfo) {
			videoInfo.setPlayLength(playlength);
		}
		return videoInfo;
	}

	/** 获取专辑下一个需要播放的视频 */
	public Map<String, VideoWrapper> getNextPlayVideoForHistory(NextPlayVideoRequest request) {
		Map<String, String> pid2Wid = Maps.newHashMap();
		// 获取所有的专辑id
		Set<String> pids = request.getPlaylist().stream().map(it -> it.getPid()).collect(Collectors.toSet());
		// 存在学员id则检查是否需要检查指定任务
		if (null != request.getSuid()) {
			DbField field = null;
			if (request.getIsSpecial()) { // 专项练习
				field = DbField.special;
			} else if (request.getIsEverday()) { // 今日任务
				field = DbField.everday;
			}
			if (null != field) { // 存在指定任务
				String commonKey = commonPlayHistoryService.getCommonKey(request.getSuid(), field);
				List<CommonPlayHistory> playHistories = commonPlayHistoryService.findPlayHistories(commonKey, pids);
				if (ArrayMapTools.isNotEmpty(playHistories)) {
					playHistories.forEach(it -> pid2Wid.put(it.getPid(), it.getWid()));
					pids.removeAll(pid2Wid.keySet());
				}
			}
		}

		Map<String, Integer> wid2Playlength = Maps.newHashMap();
		if (!pids.isEmpty() && null != request.getUid()) {
			// 获取当前专辑的播放历史
			List<UserPlayHistory> histories = userPlayHistoryService.getHistories(request.getUid(), pids);
			if (ArrayMapTools.isNotEmpty(histories)) {
				histories.forEach(it -> {
					pid2Wid.put(it.getPid(), it.getNextWid());
					wid2Playlength.put(it.getNextWid(), it.getPlayLength());
				});
				pids.removeAll(pid2Wid.keySet());
			}
		}

		// 指定任务和历史都不存在,则播放当前专辑的指定视频
		if (!pids.isEmpty()) {
			request.getPlaylist().forEach(it -> {
				if (pids.contains(it.getPid())) {
					pid2Wid.put(it.getPid(), it.getWid());
				}
			});
		}

		Map<String, VideoWrapper> pid2VideoCard = Maps.newHashMap();
		List<VideoWrapper> videoInfo = videoProxy.getVideoInfo(request.getUid(), pid2Wid.values());
		if (ArrayMapTools.isEmpty(videoInfo))
			return pid2VideoCard;
		// 补充视频的播放时长
		Map<String, VideoWrapper> wid2Video = videoInfo.stream().map(it -> {
			if (wid2Playlength.containsKey(it.getWid())) {
				it.setPlayLength(wid2Playlength.get(it.getWid()));
			} else {
				it.setPlayLength(0);
			}
			return it;
		}).collect(Collectors.toMap(it -> it.getWid(), it -> it));
		// 补充pid
		pid2Wid.forEach((tp, tw) -> {
			VideoWrapper tvw = wid2Video.get(tw);
			if (null != tvw)
				tvw = tvw.clone().setPid(tp);
			pid2VideoCard.put(tp, tvw);
		});
		return pid2VideoCard;
	}
}
