package com.jz.jar.media.proxy;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.common.utils.text.StringTools;
import com.jz.jar.media.common.ItemCard;
import com.jz.jar.media.request.NextPlayVideoRequest;
import com.jz.jar.media.request.PlaylistRequest;
import com.jz.jar.media.service.PlaylistPicsService;
import com.jz.jar.media.service.PlaylistService;
import com.jz.jar.media.service.UserCollectService;
import com.jz.jar.media.tool.AliyunBean;
import com.jz.jar.media.wrapper.PlaylistWrapper;
import com.jz.jar.media.wrapper.VideoWrapper;
import com.jz.jooq.media.tables.pojos.Playlist;
import com.jz.jooq.media.tables.pojos.PlaylistPics;

@Lazy
@Service
public class PlaylistProxy {

	@Autowired
	private PlaylistService playlistService;
	@Autowired
	private PlaylistPicsService playlistPicsService;
	@Autowired
	private UserCollectService userCollectService;
	@Autowired
	private PlaylistVideoProxy playlistVideoProxy;
	
	private Map<String, String> getPlaylistPics(List<PlaylistPics> pics) {
		if (ArrayMapTools.isEmpty(pics))
			return null;
		Map<String, String> has = Maps.newHashMap();
		pics.forEach(it -> {
			has.put(it.getSize(), AliyunBean.getScreenshotUrl(it.getUrl()));
		});
		return has;
	}
	
	public PlaylistWrapper getSinglePlaylist(PlaylistRequest req) {
		Playlist pl = playlistService.getPlaylist(req.getPid());
		if (null == pl)
			return null;
		PlaylistWrapper wp = PlaylistWrapper.of(pl);
		// 专辑图片
		wp.setPics(getPlaylistPics(playlistPicsService.getPics(req.getPid())));
		boolean isCollect = false;
		// 用户收藏
		if (StringTools.isNotEmptyAndBlank(req.getUid())) {
			isCollect = userCollectService.isCollect(req.getUid(), req.getPid());
		}
		wp.setIsCollect(isCollect);
		// 获取下一个需要播放的视频
		if (req.getWithNextVideo()) {
			wp.setVideo(playlistVideoProxy.getNextPlayVideoForHistory(req.getUid(), pl));
		}
		return wp;
	}

	public List<PlaylistWrapper> getMultiPlaylist(PlaylistRequest req) {
		List<Playlist> playlistArray = playlistService.getPlaylist(req.getPids());
		if (ArrayMapTools.isEmpty(playlistArray))
			return null;
		List<String> collectPlaylist = null;
		if (null == req.getIsCollect() && StringTools.isNotEmptyAndBlank(req.getUid())) { // 获取已经收藏的专辑
			collectPlaylist = userCollectService.getExistCollect(req.getUid(), req.getPids());
		}
		if (null == collectPlaylist)
			collectPlaylist = Lists.newArrayList();
		
		// 专辑图片
		Map<String, List<PlaylistPics>> pid2Pics = playlistPicsService.getPicsHas(req.getPids());
		
		// 下一个需要播放的视频
		Map<String, VideoWrapper> pid2VideoCard = null;
		if (req.getWithNextVideo()) {
			NextPlayVideoRequest nextRequest = NextPlayVideoRequest.of(req.getUid(), req.getSuid(), playlistArray);
			nextRequest.setIsSpecial(req.getIsSpecial()).setIsEverday(req.getIsEverday());
			pid2VideoCard = playlistVideoProxy.getNextPlayVideoForHistory(nextRequest);
		} else
			pid2VideoCard = Maps.newHashMap();

		List<PlaylistWrapper> wrappers = Lists.newArrayList();
		PlaylistWrapper wp = null;
		for (Playlist pl : playlistArray) {
			wp = PlaylistWrapper.of(pl);
			wp.setPics(this.getPlaylistPics(pid2Pics.get(pl.getPid())));
			wp.setIsCollect(null != req.getIsCollect() ? req.getIsCollect() : collectPlaylist.contains(pl.getPid()));
			if (null != req.getIsRecomm()) {
				wp.setIsRecomm(req.getIsRecomm());
			}
			wp.setVideo(pid2VideoCard.get(pl.getPid()));
			wrappers.add(wp);
		}
		return wrappers;
	}

	public List<ItemCard> getMultiPlaylistCards(PlaylistRequest req) {
		List<PlaylistWrapper> wrappers = this.getMultiPlaylist(req);
		if (null == wrappers)
			return null;
		return wrappers.stream().map(ItemCard::of).collect(Collectors.toList());
	}
}
