package com.jz.jar.media.proxy;

import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.jar.franchise.service.ClassroomService;
import com.jz.jar.franchise.service.CourseHoLessonService;
import com.jz.jar.franchise.service.FestivalAvoidService;
import com.jz.jar.franchise.service.LessonService;
import com.jz.jar.franchise.service.LessonStudentService;
import com.jz.jar.franchise.service.ManageSettingService;
import com.jz.jar.media.common.DbEnums;
import com.jz.jar.media.common.StudentClassStatus;
import com.jz.jar.media.wrapper.LessonWrapper;
import com.jz.jar.media.wrapper.TeacherWrapper;
import com.jz.jooq.franchise.tables.pojos.Classroom;
import com.jz.jooq.franchise.tables.pojos.CourseHoLesson;
import com.jz.jooq.franchise.tables.pojos.Lesson;
import com.jz.jooq.franchise.tables.pojos.LessonStudent;
import com.jz.jooq.franchise.tables.pojos.ManageSetting;

/**
 * @Title LessonProxy
 * @Package com.jz.jar.media.proxy
 * @author tangjunfeng
 * @date 2018年8月21日 下午3:51:43
 * @version V1.0
 */
@Lazy
@Service
public class LessonProxy {

	private static final SimpleDateFormat simple = new SimpleDateFormat("yyyy-MM-dd");

	@Autowired
	private LessonStudentService lessonStudentService;
	@Autowired
	private LessonService lessonService;
	@Autowired
	private ClassroomService classroomService;
	@Autowired
	private FestivalAvoidService festivalAvoidService;
	@Autowired
	private CourseHoLessonService courseHoLessonService;
	@Autowired
	private ManageSettingService manageSettingService;
	@Autowired
	private TeacherProxy teacherProxy;

	/** 获取课程表 */
	public Pair<List<LessonWrapper>, String> getTimetables(String suid, long startTime, long endTime) {
		List<LessonStudent> lessonStudent = lessonStudentService.findAllLessonForStudent(suid);
		if (ArrayMapTools.isEmpty(lessonStudent))
			return null;
		// 设置学员课程状态时使用
		Map<String, LessonStudent> schoolLessonStudent = Maps.newHashMap();
		// 查询课程使用
		Map<String, List<String>> school2Lessons = Maps.newHashMap();
		lessonStudent.forEach(it -> {
			// 获取一个学校的多个课程
			if (!school2Lessons.containsKey(it.getSchoolId()))
				school2Lessons.put(it.getSchoolId(), Lists.newArrayList());
			school2Lessons.get(it.getSchoolId()).add(it.getLessonId());
			// 根据学校和课程，方便添加上课状态
			schoolLessonStudent.put(tKey(it.getSchoolId(), it.getLessonId()), it);
		});
		// 当前时间段的课表
		List<Lesson> lessons = lessonService.findLession(school2Lessons, startTime, endTime);
		if (ArrayMapTools.isEmpty(lessons)) {
			// 验证放假信息
			String st = simple.format(new Date(startTime)), et = simple.format(new Date(endTime));
			List<String> holiday = festivalAvoidService.getHoliday(school2Lessons.keySet(), st, et);
			if (ArrayMapTools.isEmpty(holiday))
				return null;
			return Pair.of(null, holiday.get(0));
		}
		// 需要返回的wrapper信息
		List<LessonWrapper> wrappers = Lists.newArrayList();

		// 学校对于教室的信息(用于查询)
		Map<String, Collection<Integer>> s2crGet = Maps.newHashMap();
		// 学校对于教室的信息(用于封装)
		Map<String, Collection<LessonWrapper>> s2crPackage = Maps.newHashMap();

		// 老师对应课程的信息
		Map<String, Collection<LessonWrapper>> teacherHas = Maps.newHashMap();

		// 正式的课程(用于查询)
		Map<Integer, Collection<Integer>> flGet = Maps.newHashMap();
		// 正式的课程(用于封装)
		Map<String, Collection<LessonWrapper>> flPackage = Maps.newHashMap();
		
		// 学校对应班级(用于查询)
		Map<String, Collection<String>> s2clsGet = Maps.newHashMap();
		// 学校对应班级(用于包装)
		Map<String, Collection<LessonWrapper>> s2clsPackage = Maps.newHashMap();
				
		long timeMillis = System.currentTimeMillis();
		lessons.forEach(it -> {
			// 包装课程信息
			LessonWrapper wrapper = LessonWrapper.of(it);

			// 正式课
			if (it.getCourseId() > 0) {
				// 获取正式课程的名称
				putArray(flGet, it.getCourseId(), it.getLessonTime(), true);
				putArray(flPackage, tKey(it.getCourseId(), it.getLessonTime()), wrapper, false);
				
				// 还未开始上课才可以请假 > 开课15分钟则会开始自动签到
				if (it.getStartTime() > timeMillis) {
					putArray(s2clsGet, it.getSchoolId(), it.getCid(), true);
					putArray(s2clsPackage, tKey(it.getSchoolId(), it.getCid()), wrapper, false);
				}
			}
			
			// 老师id
			putArray(teacherHas, it.getTeacher(), wrapper, false);

			// 学校对应教室
			putArray(s2crGet, it.getSchoolId(), it.getClassroomId(), true);

			// 记录学校与教室信息，用map存储方便写入
			putArray(s2crPackage, tKey(it.getSchoolId(), it.getClassroomId()), wrapper, false);

			// 设置课程状态
			setClassStatus(timeMillis, it, wrapper);
			// 获取学员上课详情
			setStudentLessonStatus(schoolLessonStudent.get(tKey(it.getSchoolId(), it.getLessonId())), wrapper);
			
			wrappers.add(wrapper);
		});
		// 设置教室名称
		this.packageClassroom(s2crGet, s2crPackage);
		// 设置老师信息
		this.packageTeacher(teacherHas);
		// 设置正式课标题
		this.packageFormalLesson(flGet, flPackage);
		return Pair.of(wrappers, null);
	}

	/** 获取一个临时key */
	private String tKey(Object arg0, Object arg1) {
		return arg0 + ":" + arg1;
	}

	/** 将制定内容写入map的集合中 */
	private <K, V> void putArray(Map<K, Collection<V>> map, K key, V val, boolean isSet) {
		// 学校对应教室
		if (!map.containsKey(key)) {
			map.put(key, isSet ? Sets.newHashSet() : Lists.newArrayList());
		}
		map.get(key).add(val);
	}

	/** 设置课程的状态 */
	private void setClassStatus(long timeMillis, Lesson lesson, LessonWrapper wrapper) {
		if (lesson.getStartTime() > timeMillis) {
			wrapper.setClassStatus("即将开始");
			return;
		}
		if (lesson.getEndTime() < timeMillis) {
			wrapper.setClassStatus("已完结");
			return;
		}
		wrapper.setClassStatus("学习中");
	}

	/** 设置学员的课程状态 */
	private void setStudentLessonStatus(LessonStudent lessonStudent, LessonWrapper lesson) {
		if (null == lessonStudent)
			return;
		switch (lessonStudent.getStatus()) {
		case 0: // 未上课
			lesson.setStudentClassStatus(Arrays.asList(StudentClassStatus.prepare, StudentClassStatus.leave));
			break;
		case 1: // 已签到
		case 2: // 自动签到
			lesson.setStudentClassStatus(Arrays.asList(StudentClassStatus.review));
		default:// 旷课、请假
			lesson.setStudentClassStatus(Arrays.asList(StudentClassStatus.makeup));
			break;
		}
	}

	/** 封装老师信息 */
	private void packageTeacher(Map<String, Collection<LessonWrapper>> teacherHas) {
		Map<String, TeacherWrapper> teachers = teacherProxy.getTeachers(teacherHas.keySet());
		if (ArrayMapTools.isEmpty(teachers))
			return;
		teachers.forEach((id, t) -> {
			if (teacherHas.containsKey(id)) {
				teacherHas.get(id).forEach(w -> w.setTeacher(t));
			}
		});
	}

	/** 封装教室名称 */
	private void packageClassroom(Map<String, Collection<Integer>> s2crGet, Map<String, Collection<LessonWrapper>> s2crPackage) {
		List<Classroom> classrooms = classroomService.findClassrooms(s2crGet);
		if (ArrayMapTools.isEmpty(classrooms))
			return;
		classrooms.forEach(it -> {
			String key = tKey(it.getSchoolId(), it.getClassroomId());
			if (s2crPackage.containsKey(key)) {
				s2crPackage.get(key).forEach(ws -> ws.setClassroom(it.getName()));
			}
		});
	}

	/** 封装正式课信息 */
	private void packageFormalLesson(Map<Integer, Collection<Integer>> flGet, Map<String, Collection<LessonWrapper>> flPackage) {
		List<CourseHoLesson> hoLession = courseHoLessonService.findHoLession(DbEnums.playabc.name(), flGet);
		if (ArrayMapTools.isEmpty(hoLession))
			return;
		hoLession.forEach(it -> {
			String key = tKey(it.getCourseId(), it.getLessonTime());
			if (flPackage.containsKey(key)) {
				flPackage.get(key).forEach(ws -> ws.setLessonTitle(it.getLessonTitle()));
			}
		});
	}
	
	/** 获取学校配置的最大可请假天数 */
	private Map<String, Integer> getLeaveNum(Collection<String> schools) {
		List<ManageSetting> leaveNum = manageSettingService.getLeaveNum(schools);
		if (ArrayMapTools.isEmpty(leaveNum))
			return Maps.newHashMap();
		return leaveNum.stream().collect(Collectors.toMap(it -> it.getSchoolId(), it -> it.getLeaveMaxNum()));
	}
}
