package com.jz.jar.media.proxy;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.common.utils.text.StringTools;
import com.jz.jar.media.service.DayPlaySettingService;
import com.jz.jar.media.service.MonthPlaySettingService;
import com.jz.jar.media.service.PlaylistRelationService;
import com.jz.jar.media.service.StudentDayPlayService;
import com.jz.jar.media.service.StudentMonthPlayService;
import com.jz.jar.media.wrapper.StudentMonthPlayWrapper;
import com.jz.jooq.media.tables.pojos.Level;
import com.jz.jooq.media.tables.pojos.MonthPlaySetting;
import com.jz.jooq.media.tables.pojos.StudentMonthPlay;
import com.jz.jooq.media.tables.pojos.Topic;

@Lazy
@Service
public class StudentPlayProxy {

	private static final SimpleDateFormat yearMonth = new SimpleDateFormat("yyyyMM");
	private static final SimpleDateFormat yearMonthDay = new SimpleDateFormat("yyyyMMDD");

	@Autowired
	private MonthPlaySettingService monthPlaySettingService;
	@Autowired
	private StudentMonthPlayService studentMonthPlayService;
	@Autowired
	private PlaylistRelationService playlistRelationService;
	@Autowired
	private DayPlaySettingService dayPlaySettingService;
	@Autowired
	private StudentDayPlayService studentDayPlayService;
	@Autowired
	private StudentLevelProxy studentLevelProxy;
	@Autowired
	private TopicProxy topicProxy;
	
	/** 获取学员月观看配置 */
	public List<MonthPlaySetting> getStudentMonthPlaySetting(Level level) {
		if (null == level) // 年龄匹配失败
			return null;
		// 获取观看配置信息
		return monthPlaySettingService.findSettingForLid(level.getLid());
	}
	
	/** 获取学员月观看配置 */
	public List<MonthPlaySetting> getStudentMonthPlaySetting(String uid, String suid) {
		// 获取观看配置信息
		return this.getStudentMonthPlaySetting(studentLevelProxy.getLevelForAge(uid, suid));
	}
	
	/** 获取当前月数字 */
	public String getMonth() {
		return Calendar.getInstance().get(Calendar.MONTH) + 1 + "月";
	}
	
	/** 获取学员月观看详情 */
	public List<StudentMonthPlayWrapper> getStudentPlayInfo(String uid, String suid) {
		// 获取观看配置信息
		return this.getStudentPlayInfo(uid, suid, this.getStudentMonthPlaySetting(uid, suid));
	}
	
	/** 获取学员月观看详情 */
	public List<StudentMonthPlayWrapper> getStudentPlayInfo(String uid, String suid, List<MonthPlaySetting> settings) {
		Map<String, Integer> playCid2Minutes = Maps.newHashMap(), taskCid2Minutes = Maps.newHashMap();
		if (ArrayMapTools.isEmpty(settings)) { // 获取所有的分类
			taskCid2Minutes.putAll(
					topicProxy.getTopics().stream().collect(Collectors.toMap(it-> it.getCid(), it-> 0))
					);
		} else { // 获取配置的月播放信息
			taskCid2Minutes.putAll(
					settings.stream().collect(Collectors.toMap(it -> it.getCid(), it -> it.getMinutes()))
					);
		}
		// 根据当前月获取学员观看信息
		String currentMonth = yearMonth.format(Calendar.getInstance().getTime());
		String tempUid = StringTools.ternary(suid, uid);
		List<StudentMonthPlay> plays = studentMonthPlayService.getStudentPlayForMonth(currentMonth, tempUid);
		if (ArrayMapTools.isNotEmpty(plays)) {
			// 将学员的观看历史秒数转换成分钟
			playCid2Minutes.putAll(
					plays.stream().collect(Collectors.toMap(it -> it.getCid(), it -> it.getSeconds() / 60))
					);
		}
		// 有计划就使用计划中的分类,无则使用学员现有的配置
		List<String> tids = Lists.newArrayList(taskCid2Minutes.keySet());
		if (tids.isEmpty()) {
			tids.addAll(playCid2Minutes.keySet());
		}
		List<StudentMonthPlayWrapper> wrappers = Lists.newArrayList();
		for (Topic tc : topicProxy.getTopics(tids)) {
			wrappers.add(StudentMonthPlayWrapper.of(tc.getName())
					.setTaskMinutes(ArrayMapTools.getInteger(taskCid2Minutes, tc.getCid()))
					.setPlayMinutes(ArrayMapTools.getInteger(playCid2Minutes, tc.getCid(), 0)));
		}
		return wrappers;
	}
	
	/** 保存月观看任务 */
	public void saveStudentMonthPlay(String puid, String suid, String pid, int seconds, Level level) {
		// 月观看任务
		List<String> topics = null;
		// 获取当前学员的月观看任务
		List<MonthPlaySetting> setting = this.getStudentMonthPlaySetting(level);
		if (ArrayMapTools.isNotEmpty(setting)) {
			topics = setting.stream().map(it -> it.getCid()).collect(Collectors.toList());
		} else { // 如果不是学员则记录所有的观看记录
			topics = playlistRelationService.getTopicsForPlaylist(pid);
		}
		if (ArrayMapTools.isEmpty(topics))
			return;
		// 保存月观看任务数据
		String currentMonth = yearMonth.format(Calendar.getInstance().getTime());
		studentMonthPlayService.saveMonthPlay(currentMonth, StringTools.ternary(suid, puid), seconds, topics);
	}
	
	/** 获取每日任务的专辑列表 */
	public List<String> getTodayTaskPlaylist(String lid) {
		String today = yearMonthDay.format(Calendar.getInstance().getTime());
		return dayPlaySettingService.getTodayTaskPlaylist(today, lid);
	}
	
	/** 是否进入任务的专辑 */
	public boolean isTodayTaskPlaylist(String lid, String pid) {
		String today = yearMonthDay.format(Calendar.getInstance().getTime());
		return dayPlaySettingService.isTodayTaskPlaylist(today, lid, pid);
	}
	
	/** 保存学员今日观看信息 */
	public void saveStudentTodayPlay(String suid, String pid, int targetCnt, int playCnt) {
		String today = yearMonthDay.format(Calendar.getInstance().getTime());
		studentDayPlayService.saveTodayPlay(today, suid, pid, targetCnt, playCnt);
	}
	
	/** 获取当前学员今日已经完成的专辑id */
	public List<String> getTodayCompletePlaylist(String suid, Collection<String> pids) {
		String today = yearMonthDay.format(Calendar.getInstance().getTime());
		return studentDayPlayService.getTodayCompletePlaylist(today, suid, pids);
	}
}
