package com.jz.jar.media.cache;

import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import com.google.common.collect.Maps;
import com.jz.common.utils.collection.ArrayMapTools;

import redis.clients.jedis.JedisCluster;
import redis.clients.jedis.Tuple;

/**
 * @Title SpecialPracticeCache
 * @Package com.jz.jar.media.cache
 * @author tangjunfeng
 * @date 2018年8月13日 下午5:25:48
 * @version V1.0
 */
@Lazy
@Component
public class SpecialPracticeCache {

	private static final Logger logger = LoggerFactory.getLogger(SpecialPracticeCache.class);
	// 1h
	private static final int seconds = 3600;

	private static final String EMPTY = "empty";

	@Autowired
	private JedisCluster jedisCluster;

	private String getKey(String suid, String cid, boolean isFinished) {
		int finishedInt = isFinished ? 1 : 0;
		return new StringBuffer().append("special:practice:playlist:").append(suid).append(":").append(cid).append(":")
				.append(finishedInt).toString();
	}

	/** 计算redis结束位置 */
	private static int getEndIndex(int start, int size) {
		return size > 0 ? start + size - 1 : -1;
	}

	public void setSpecialPlaylist(String suid, String cid, Map<String, Integer> playlist2Percent, boolean isFinished) {
		try {
			Map<String, Double> pid2Score = Maps.newHashMap();
			if (ArrayMapTools.isNotEmpty(playlist2Percent)) {
				playlist2Percent.forEach((k, v) -> pid2Score.put(k, v.doubleValue()));
			} else {
				pid2Score.put(EMPTY, -1d);
			}
			String key = getKey(suid, cid, isFinished);
			jedisCluster.zadd(key, pid2Score);
			jedisCluster.expire(key, seconds);
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		}
	}

	public Map<String, Integer> getSpecialPlaylist(String suid, String cid, boolean isFinished, int start, int size) {
		try {
			String key = getKey(suid, cid, isFinished);
			System.out.println(key);
			Set<Tuple> tuples = jedisCluster.zrangeWithScores(key, start, getEndIndex(start, size));
			if (ArrayMapTools.isEmpty(tuples)) {
				if (jedisCluster.exists(key))
					return Maps.newHashMap();
				return null;
			}
			Map<String, Integer> playlist2Percent = Maps.newLinkedHashMap();
			for (Tuple tuple : tuples) {
				if (EMPTY.equals(tuple.getElement()))
					return Maps.newHashMap(); // 当前列表是空的
				playlist2Percent.put(tuple.getElement(), (int) tuple.getScore());
			}
			return playlist2Percent;
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			return null;
		}
	}
	
	public int countSpecialPlaylist(String suid, String cid, boolean isFinished) {
		try {
			String key = getKey(suid, cid, isFinished);
			Long zrank = jedisCluster.zrank(key, EMPTY);
			if (null != zrank) // 验证当前列表是否是个空列表
				return 0;
			// 非空列表
			Long total = jedisCluster.zcard(key);
			if (null != total)
				return total.intValue();
			return 0;
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			return -1;
		}
	}
}
