package com.jz.gymchina.jar.resources.repository;

import java.util.Collection;
import java.util.List;

import org.jooq.Condition;
import org.jooq.TableOnConditionStep;
import org.jooq.UpdateSetMoreStep;
import org.jooq.impl.DSL;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;

import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.common.utils.text.StringTools;
import com.jz.gymchina.jar.resources.extend.GymUserInfoJoinUserGroup;
import com.jz.jooq.gymchina.resources.Tables;
import com.jz.jooq.gymchina.resources.tables.pojos.GymUserInfo;
import com.jz.jooq.gymchina.resources.tables.records.GymUserInfoRecord;

@Lazy
@Repository
public class GymUserInfoRepository extends ResourceBaseRepository {

	private static com.jz.jooq.gymchina.resources.tables.GymUserInfo UI = Tables.GYM_USER_INFO;
	private static com.jz.jooq.gymchina.resources.tables.UserGroup UG = Tables.USER_GROUP;

	public boolean existUsername(String username) {
		return resourcesCtx.fetchExists(UI, UI.USERNAME.eq(username));
	}

	public GymUserInfo getUserForLogin(String username, String password) {
		return resourcesCtx.select(UI.UID, UI.USERNAME, UI.NICKNAME, UI.ENABLE, UI.IS_ADMIN).from(UI)
				.where(UI.USERNAME.eq(username).and(UI.PASSWORD.eq(password))).fetchAnyInto(GymUserInfo.class);
	}
	
	public GymUserInfo getUser(String uid) {
		return resourcesCtx.selectFrom(UI).where(UI.UID.eq(uid)).fetchAnyInto(GymUserInfo.class);
	}
	
	public void addUserInfo(String uid, String username, String password, String nickname, String email) {
		GymUserInfoRecord userInfo = new GymUserInfoRecord();
		userInfo.setUid(uid);
		userInfo.setUsername(username);
		userInfo.setPassword(password);
		userInfo.setNickname(nickname);
		userInfo.setEmail(email);
		userInfo.setIsAdmin(0); // 非管理员
		userInfo.setLastUpdate(System.currentTimeMillis());
		userInfo.setEnable(1); // 有效的账户
		resourcesCtx.insertInto(UI).set(userInfo).execute();
	}

	public void updatePassword(String uid, String password) {
		resourcesCtx.update(UI).set(UI.PASSWORD, password).set(UI.LAST_UPDATE, System.currentTimeMillis())
				.where(UI.UID.eq(uid)).execute();
	}

	public void updateUserInfo(String uid, String nickname, String email) {
		UpdateSetMoreStep<GymUserInfoRecord> moreStep = resourcesCtx.update(UI).set(UI.LAST_UPDATE,
				System.currentTimeMillis());
		if (StringTools.isNotEmptyAndBlank(nickname))
			moreStep = moreStep.set(UI.NICKNAME, nickname);
		if (StringTools.isNotEmptyAndBlank(email))
			moreStep = moreStep.set(UI.EMAIL, email);
		moreStep.where(UI.UID.eq(uid)).execute();
	}

	public void updateUserEnableInfo(String uid, int enable) {
		resourcesCtx.update(UI).set(UI.ENABLE, enable).set(UI.LAST_UPDATE, System.currentTimeMillis())
				.where(UI.UID.eq(uid)).execute();
	}

	public boolean isAdmin(String uid) {
		return resourcesCtx.fetchExists(UI, UI.UID.eq(uid).and(UI.ENABLE.eq(1)).and(UI.IS_ADMIN.eq(1)));
	}

	public List<String> getForbiddenUsers() {
		return resourcesCtx.select(UI.UID).from(UI).where(UI.ENABLE.eq(0)).fetchInto(String.class);
	}

	private TableOnConditionStep getJoinTables() {
		return UI.leftOuterJoin(UG).on(UI.UID.eq(UG.UID));
	}

	private Condition getWhereCondition(String name, Integer enable, Collection<String> gids) {
		Condition condition = DSL.trueCondition();
		if (StringTools.isNotEmptyAndBlank(name))
			condition = condition.and(DSL.or(UI.USERNAME.like("%" + name + "%"), UI.NICKNAME.like("%" + name + "%")));
		if (null != enable)
			condition = condition.and(UI.ENABLE.eq(enable));
		if (ArrayMapTools.isNotEmpty(gids))
			condition = condition.and(UG.GID.in(gids));
		return condition;
	}

	public int countUsers(String name, Integer enable, Collection<String> gids) {
		return resourcesCtx.fetchCount(resourcesCtx.select(UI.UID).from(getJoinTables())
				.where(getWhereCondition(name, enable, gids)).groupBy(UI.UID));
	}

	public List<GymUserInfoJoinUserGroup> getUsers(String name, Integer enable, Collection<String> gids, int start,
			int size) {
		return resourcesCtx
				.select(UI.UID, UI.USERNAME, UI.NICKNAME, UI.EMAIL, UI.ENABLE, UI.IS_ADMIN, DSL.groupConcat(UG.GID).as("gids"))
				.from(getJoinTables()).where(getWhereCondition(name, enable, gids)).groupBy(UI.UID)
				.orderBy(UI.LAST_UPDATE.desc()).limit(start, size).fetchInto(GymUserInfoJoinUserGroup.class);
	}
}
