package com.jz.jar.franchise.repository;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import org.jooq.Condition;
import org.jooq.impl.DSL;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;

import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.common.utils.text.StringTools;
import com.jz.jooq.franchise.Tables;
import com.jz.jooq.franchise.tables.pojos.School;

@Lazy
@Repository
public class SchoolRepository extends FranchiseBaseRepository {

	private static final com.jz.jooq.franchise.tables.School SC = Tables.SCHOOL;

	public School getSchool(String id) {
		return franchiseCtx.selectFrom(SC).where(SC.ID.eq(id).and(SC.STATUS.eq(1))).fetchAnyInto(School.class);
	}

	public List<School> getSimpleSchool(Collection<String> ids) {
		return franchiseCtx.select(SC.ID, SC.CENTER_NAME, SC.CENTER_ADDRESS, SC.PHONE).from(SC)
				.where(SC.ID.in(ids).and(SC.STATUS.eq(1))).fetchInto(School.class);
	}

	private Condition getSchoolCondition(String brand, String prov, String city, boolean isReservation) {
		// 预约的学校只展示 code >= 100 的学校
		Condition condition = SC.BRAND_ID.eq(brand).and(SC.STATUS.eq(1)).and(SC.CODE.ge(100));
		if (StringTools.isNotEmpty(prov))
			condition = condition.and(SC.PROV.eq(prov));
		if (StringTools.isNotEmpty(city))
			condition = condition.and(SC.CITY.eq(city));
		if (isReservation) // 非预约试听可展示所有的学校
			condition = condition.and(SC.CENTER_NAME.isNotNull()).and(SC.CENTER_NAME.ne(""));
		return condition;
	}

	/** city_level字段用作表示学校数量（不想添加字段） */
	public List<School> getSchoolAreas(String brand, boolean isReservation) {
		return franchiseCtx.select(SC.PROV, SC.CITY, DSL.count(DSL.one()).as(SC.CITY_LEVEL)).from(SC)
				.where(getSchoolCondition(brand, null, null, isReservation)).groupBy(SC.PROV, SC.CITY)
				.fetchInto(School.class);
	}

	public List<School> getSchools(String brand, String prov, String city, boolean isReservation) {
		return franchiseCtx.select(SC.ID, SC.CENTER_NAME, SC.CENTER_ADDRESS, SC.PHONE).from(SC)
				.where(getSchoolCondition(brand, prov, city, isReservation)).orderBy(SC.CODE.desc())
				.fetchInto(School.class);
	}

	public String getSchoolName(String id) {
		return franchiseCtx.select(SC.CENTER_NAME).from(SC).where(SC.ID.eq(id).and(SC.STATUS.eq(1)))
				.fetchAnyInto(String.class);
	}

	public List<String> getSchoolIdForBrand(String brand) {
		return franchiseCtx.select(SC.ID).from(SC).where(SC.BRAND_ID.eq(brand).and(SC.STATUS.eq(1)))
				.fetchInto(String.class);
	}

	public List<String> getSchoolIdsByName(String name) {
		return franchiseCtx.select(SC.ID).from(SC).where(SC.NAME.like("%" + name + "%").and(SC.TYPE.eq(2)))
				.fetchInto(String.class);
	}

	public int cntJoinSchool(Collection<String> filterSchoolIds) {
		return franchiseCtx.fetchCount(SC, SC.TYPE.eq(2)
				.and(ArrayMapTools.isNotEmpty(filterSchoolIds) ? SC.ID.in(filterSchoolIds) : DSL.trueCondition()));
	}

	public List<School> getJoinSchools(Collection<String> filterSchoolIds, int start, int size) {
		return franchiseCtx.select(SC.ID, SC.BRAND_ID, SC.CODE, SC.NAME).from(SC)
				.where(SC.TYPE.eq(2).and(
						ArrayMapTools.isNotEmpty(filterSchoolIds) ? SC.ID.in(filterSchoolIds) : DSL.trueCondition()))
				.orderBy(SC.BRAND_ID.asc(), SC.CODE.asc()).limit(start, size).fetchInto(School.class);
	}

	public List<School> mutiGetSimpleSchools(Set<String> schoolIds) {
		return franchiseCtx.select(SC.ID, SC.BRAND_ID, SC.CODE, SC.NAME).where(SC.ID.in(schoolIds))
				.fetchInto(School.class);
	}
}
