package com.jz.jar.franchise.repository;

import java.util.Collection;
import java.util.List;

import org.jooq.Condition;
import org.jooq.impl.DSL;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;

import com.jz.jar.franchise.enums.ContractParentSignStatus;
import com.jz.jooq.franchise.Tables;
import com.jz.jooq.franchise.tables.pojos.Contract;
import com.jz.jooq.franchise.tables.pojos.ContractDetail;

/**
 * @author 连杰
 */
@Lazy
@Repository
public class ContractRepository extends FranchiseBaseRepository {

	private static final com.jz.jooq.franchise.tables.Contract C = Tables.CONTRACT;
	private static final com.jz.jooq.franchise.tables.ContractDetail CD = Tables.CONTRACT_DETAIL;
	private static final com.jz.jooq.franchise.tables.ContractParentSign CPS = Tables.CONTRACT_PARENT_SIGN;

	public Contract getContract(String contractId, String schoolId) {
		return franchiseCtx.selectFrom(C).where(C.CONTRACT_ID.eq(contractId).and(C.SCHOOL_ID.eq(schoolId)))
				.fetchAnyInto(Contract.class);
	}

	public Contract getContractForBranch(String brandId, String contractId) {
		return franchiseCtx.selectFrom(C).where(C.CONTRACT_ID.eq(contractId).and(C.BRAND_ID.eq(brandId)))
				.fetchAnyInto(Contract.class);
	}

	public ContractDetail getContractDetail(String contractId, String schoolId) {
		return franchiseCtx.selectFrom(CD).where(CD.CONTRACT_ID.eq(contractId).and(CD.SCHOOL_ID.eq(schoolId)))
				.fetchAnyInto(ContractDetail.class);
	}

	public List<ContractDetail> mutiGetContractGiftCompany(Collection<String> contractIds) {
		return franchiseCtx.select(CD.CONTRACT_ID, CD.GIFT_NAME, CD.COMPANY_ID, CD.TERM_ID).from(CD)
				.where(CD.CONTRACT_ID.in(contractIds)).fetchInto(ContractDetail.class);
	}

	public List<Contract> getUserContractList(String brandId, String puid) {
		return franchiseCtx.selectFrom(C).where(C.PUID.eq(puid).and(C.STATUS.gt(0)).and(C.BRAND_ID.eq(brandId)))
				.orderBy(C.CREATED.desc()).fetchInto(Contract.class);
	}

	public List<Contract> getPayedContractList(String brandId, String puid) {
		return franchiseCtx.selectFrom(C).where(C.PUID.eq(puid).and(C.STATUS.eq(1)).and(C.BRAND_ID.eq(brandId)))
				.orderBy(C.PAY_TIME.desc()).fetchInto(Contract.class);
	}

//	public boolean isHaveContract(String brandId, String puid) {
//		return franchiseCtx.fetchExists(C, this.getContractCondition(brandId, puid));
//	}

	public List<String> getHaveContractSchoolIds(String brandId, String puid) {
		return franchiseCtx.select(C.SCHOOL_ID).from(C).where(this.getContractCondition(brandId, puid))
				.groupBy(C.SCHOOL_ID).fetchInto(String.class);
	}

	public boolean isHaveContractInSchool(String schoolId, String puid) {
		return franchiseCtx.fetchExists(C, C.SCHOOL_ID.eq(schoolId).and(C.PUID.eq(puid))
				.and(DSL.or(C.STATUS.in(1, 50), C.STATUS.ge(20).and(C.STATUS.le(39)))));
	}

	public List<String> getContractedSchoolIds(String brandId, String puid) {
		return franchiseCtx.select(C.SCHOOL_ID).from(C).where(this.getContractCondition(brandId, puid))
				.groupBy(C.SCHOOL_ID).fetchInto(String.class);
	}

	private Condition getContractCondition(String brandId, String puid) {
		// status in(1,50) or (status>=20 and status<=39)
		Condition condition = C.PUID.eq(puid).and(C.BRAND_ID.eq(brandId));
		condition = condition.and(DSL.or(C.STATUS.in(1, 50), C.STATUS.ge(20).and(C.STATUS.le(39))));
		return condition;
	}

	public boolean isOwnedContract(String branchId, String puid) {
		return franchiseCtx.fetchExists(C, C.PUID.eq(puid).and(C.BRAND_ID.eq(branchId)));
	}

	public Contract getSimpleContract(String contractId) {
		return franchiseCtx.select(C.START_TIME).from(C).where(C.CONTRACT_ID.eq(contractId))
				.fetchAnyInto(Contract.class);
	}

	public String getParentSignContent(String contractId) {
		return franchiseCtx.select(CPS.CONTENT).from(CPS).where(CPS.CONTRACT_ID.eq(contractId))
				.fetchAnyInto(String.class);
	}

	public int cntWaitParentSignForAppOpen(String brandId, String puid) {
		return franchiseCtx.fetchCount(C,
				C.PUID.eq(puid).and(C.STATUS.in(10, 12, 14, 16, 18)).and(C.BRAND_ID.eq(brandId))
						.and(C.PARENT_SIGN_STATUS.eq(ContractParentSignStatus.waitSignAffect.code)));
	}

	public List<String> getSchoolIds(String brandId, String puid) {
		return franchiseCtx.select(C.SCHOOL_ID).from(C).where(C.PUID.eq(puid).and(C.BRAND_ID.eq(brandId)))
				.fetchInto(String.class);
	}

	public boolean checkIsHaveContractAllBrand(String puid) {
		return franchiseCtx.fetchExists(C, C.PUID.eq(puid));
	}

	public String getContractedSchoolId(String brandId, String puid) {
		return franchiseCtx.select(C.SCHOOL_ID).from(C).where(this.getContractCondition(brandId, puid))
				.orderBy(C.CREATED.desc()).limit(1).fetchAnyInto(String.class);
	}

	public String getSuidRecentSchoolId(Collection<String> allContractIds, String brand) {
		return franchiseCtx.select(C.SCHOOL_ID).from(C)
				.where(C.CONTRACT_ID.in(allContractIds).and(C.BRAND_ID.eq(brand)).and(C.PAY_TIME.isNotNull()))
				.orderBy(C.CREATED.desc()).limit(1).fetchAnyInto(String.class);
	}

	public Contract getContractById(String contractId) {
		return franchiseCtx.selectFrom(C).where(C.CONTRACT_ID.eq(contractId)).fetchAnyInto(Contract.class);
	}

}
