package com.jz.jar.franchise.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Condition;
import org.jooq.impl.DSL;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.jooq.franchise.Tables;
import com.jz.jooq.franchise.tables.pojos.StudentSchoolContract;
import com.jz.jooq.franchise.tables.pojos.StudentSchoolExtra;
import com.jz.jooq.franchise.tables.records.StudentSchoolContractRecord;

@Lazy
@Repository
public class StudentSchoolContractRepository extends FranchiseBaseRepository implements
		FranchiseConditionRepository<StudentSchoolContractRecord> {

	private static final com.jz.jooq.franchise.tables.StudentSchoolContract SSC = Tables.STUDENT_SCHOOL_CONTRACT;
	private static final com.jz.jooq.franchise.tables.StudentSchoolExtra SSE = Tables.STUDENT_SCHOOL_EXTRA;

	public Pair<Integer, Integer> getLessionCount(String suid, Collection<String> schoolIds) {
		List<Map<String, Object>> maps = franchiseCtx
				.select(DSL.sum(SSC.TOTAL_OFFICAL_LESSON).as("total"),
						DSL.sum(SSC.CONSUME_OFFICAL_LESSON).as("consume")).from(SSC)
				.where(SSC.SUID.eq(suid).and(SSC.SCHOOL_ID.in(schoolIds)).and(SSC.STATUS.ne(-1))).fetchMaps();
		if (ArrayMapTools.isEmpty(maps))
			return Pair.of(0, 0);
		Map<String, Object> has = maps.get(0);
		if (ArrayMapTools.isEmpty(has) || !ArrayMapTools.containsKeys(has, "total", "consume"))
			return Pair.of(0, 0);
		return Pair.of(ArrayMapTools.getInteger(has, "total", 0), ArrayMapTools.getInteger(has, "consume", 0));
	}

	public Map<String, Pair<Integer, Integer>> getLessionCount(Collection<String> suids, Collection<String> schoolIds) {
		List<Map<String, Object>> fetchMaps = franchiseCtx
				.select(SSC.SUID, DSL.sum(SSC.TOTAL_OFFICAL_LESSON).as("total"),
						DSL.sum(SSC.CONSUME_OFFICAL_LESSON).as("consume")).from(SSC)
				.where(SSC.SUID.in(suids).and(SSC.SCHOOL_ID.in(schoolIds)).and(SSC.STATUS.ne(-1))).fetchMaps();
		if (ArrayMapTools.isEmpty(fetchMaps))
			return Maps.newHashMap();
		Map<String, Pair<Integer, Integer>> result = Maps.newHashMap();
		for (Map<String, Object> has : fetchMaps) {
			if (ArrayMapTools.isEmpty(has) || !ArrayMapTools.containsKeys(has, "suid", "total", "consume"))
				continue;
			result.put(MapUtils.getString(has, "suid"),
					Pair.of(MapUtils.getInteger(has, "total"), MapUtils.getInteger(has, "consume")));
		}
		return result;
	}

	/** 获取当前合同已经请假的天数 */
	public List<StudentSchoolContract> getSimpleStudentSchoolContracts(String suid,
			Map<String, Collection<String>> school2ContractId) {
		Condition condition = getOrWhereCondition(SSC.SCHOOL_ID, SSC.CONTRACT_ID, school2ContractId);
		condition = SSC.SUID.eq(suid).and(condition);
		return franchiseCtx.select(SSC.SCHOOL_ID, SSC.CONTRACT_ID, SSC.LEAVE_NUM, SSC.MAX_LEAVE_NUM).from(SSC)
				.where(condition).fetchInto(StudentSchoolContract.class);
	}

	public StudentSchoolContract getStudentSchoolContract(String suid, String school, String contractId) {
		return franchiseCtx.selectFrom(SSC)
				.where(SSC.SUID.eq(suid).and(SSC.SCHOOL_ID.eq(school)).and(SSC.CONTRACT_ID.eq(contractId)))
				.fetchAnyInto(StudentSchoolContract.class);
	}

	public void addLeaveNum(String suid, String school, String contractId, int addCnt) {
		franchiseCtx.update(SSC).set(SSC.LEAVE_NUM, SSC.LEAVE_NUM.add(addCnt))
				.where(SSC.SUID.eq(suid).and(SSC.SCHOOL_ID.eq(school)).and(SSC.CONTRACT_ID.eq(contractId))).execute();
	}

	public List<StudentSchoolContract> getSSCByContractId(String schoolId, String contractId) {
		return franchiseCtx.selectFrom(SSC).where(SSC.SCHOOL_ID.eq(schoolId).and(SSC.CONTRACT_ID.eq(contractId)))
				.fetchInto(StudentSchoolContract.class);
	}

	public List<StudentSchoolContract> mutiGetSSCByContractIds(Collection<String> contractIds) {
		return franchiseCtx.selectFrom(SSC).where(SSC.CONTRACT_ID.in(contractIds))
				.fetchInto(StudentSchoolContract.class);
	}

	public List<Map<String, Object>> mutiCalContractConsumes(List<String> contractIds) {
		return franchiseCtx
				.select(SSC.CONTRACT_ID.as("contractId"),
						DSL.sum(SSC.CONSUME_OFFICAL_LESSON).as("consumeOfficalLesson"),
						DSL.sum(SSC.NO_SCHEDULE_OFFICAL_LESSON).as("noScheduleOfficalLesson")).from(SSC)
				.where(SSC.CONTRACT_ID.in(contractIds)).groupBy(SSC.CONTRACT_ID).fetchMaps();
	}

	public List<String> getIsContinueContract(List<String> contractIds) {
		return franchiseCtx.select(SSC.CONTRACT_ID).from(SSC)
				.where(SSC.CONTRACT_ID.in(contractIds).and(SSC.CONSUME_OFFICAL_LESSON.lt(SSC.TOTAL_OFFICAL_LESSON)))
				.groupBy(SSC.CONTRACT_ID).fetchInto(String.class);
	}

	public boolean dealRollbackOfficalLesson(String suid, String schoolId, String contractId, int rollbackNum) {
		Preconditions.checkArgument(rollbackNum > 0, "scheduleNum>0");
		return franchiseCtx.update(SSC)
				.set(SSC.NO_SCHEDULE_OFFICAL_LESSON, SSC.NO_SCHEDULE_OFFICAL_LESSON.add(rollbackNum))
				.where(SSC.SUID.eq(suid).and(SSC.SCHOOL_ID.eq(schoolId)).and(SSC.CONTRACT_ID.eq(contractId))).execute() > 0;
	}

	public boolean dealRollbackExtraLesson(String suid, String schoolId, int extraId, int rollbackNum) {
		Preconditions.checkArgument(rollbackNum > 0, "rollbackNum>0");
		return franchiseCtx.update(SSE).set(SSE.NO_SCHEDULE, SSE.NO_SCHEDULE.add(rollbackNum))
				.where(SSE.ID.eq(extraId).and(SSE.SUID.eq(suid)).and(SSE.SCHOOL_ID.eq(schoolId))).execute() > 0;
	}

	public List<StudentSchoolExtra> mutiGetContractExtras(Collection<String> contractIds) {
		return franchiseCtx.selectFrom(SSE).where(SSE.CONTRACT_ID.in(contractIds)).fetchInto(StudentSchoolExtra.class);
	}

	/** 获取学员的课时数 */
	public Map<String, List<StudentSchoolContract>> getStudentLessonCount(Collection<String> suids,
			Collection<String> schoolIds) {
		return franchiseCtx.select(SSC.SUID, SSC.SCHOOL_ID, SSC.TOTAL_OFFICAL_LESSON, SSC.CONSUME_OFFICAL_LESSON)
				.from(SSC).where(SSC.SUID.in(suids).and(SSC.SCHOOL_ID.in(schoolIds)).and(SSC.STATUS.ge(0)))
				.fetchGroups(SSC.SUID, StudentSchoolContract.class);
	}

	/** 获取学员的赠课时数 */
	public Map<String, List<StudentSchoolExtra>> getStudentExtraLessonCount(Map<String, List<String>> suid2SchoolIds) {
		List<Condition> conditions = Lists.newArrayList();
		for (Entry<String, List<String>> entry : suid2SchoolIds.entrySet()) {
			conditions.add(SSE.SUID.eq(entry.getKey()).and(SSE.SCHOOL_ID.in(entry.getValue())));
		}
		return franchiseCtx.select(SSE.SUID, SSE.TOTAL, SSE.CONSUME).from(SSE)
				.where(DSL.or(conditions).and(SSE.STATUS.eq(1))).fetchGroups(SSE.SUID, StudentSchoolExtra.class);
	}

	public boolean isExistValidContract(String suid) {
		return franchiseCtx.fetchExists(SSC, SSC.SUID.eq(suid).and(SSC.STATUS.ge(0)));
	}

	public boolean isExistContract(String suid) {
		return franchiseCtx.fetchExists(SSC, SSC.SUID.eq(suid));
	}
}
