package com.jz.jar.franchise.service;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.jar.franchise.repository.ContractPostponeRepository;
import com.jz.jar.franchise.repository.ContractRepository;
import com.jz.jar.franchise.repository.ContractTermsRepository;
import com.jz.jar.franchise.repository.StudentInfoRepository;
import com.jz.jar.franchise.repository.StudentSchoolContractRepository;
import com.jz.jar.franchise.wrapper.ContractPostponeWrapper;
import com.jz.jar.franchise.wrapper.ContractStudentWrapper;
import com.jz.jar.franchise.wrapper.ContractWrapper;
import com.jz.jar.franchise.wrapper.CourseWrapper;
import com.jz.jar.franchise.wrapper.UserSimpleWrapper;
import com.jz.jooq.franchise.tables.pojos.Contract;
import com.jz.jooq.franchise.tables.pojos.ContractDetail;
import com.jz.jooq.franchise.tables.pojos.ContractPostpone;
import com.jz.jooq.franchise.tables.pojos.School;
import com.jz.jooq.franchise.tables.pojos.StudentInfo;
import com.jz.jooq.franchise.tables.pojos.StudentSchoolContract;

@Lazy
@Service
public class ContractService {

	private Logger logger = LoggerFactory.getLogger(ContractService.class);

	@Autowired
	private ContractRepository contractRepository;
	@Autowired
	private StudentSchoolContractRepository studentSchoolContractRepository;
	@Autowired
	private FUserInfoService fUserInfoService;
	@Autowired
	private CourseService courseService;
	@Autowired
	private CoursePackService coursePackService;
	@Autowired
	private ParentInfoService parentInfoService;
	@Autowired
	private StudentInfoRepository studentInfoRepository;
	@Autowired
	private ContractPostponeRepository contractPostponeRepository;
	@Autowired
	private ContractTermsRepository contractTermsRepository;

	public Contract getContract(String contractId) {
		return contractRepository.getContract(contractId);
	}

	public ContractWrapper getContractDetailWrapper(String contractId, String schoolId, School school) {
		Contract contract = contractRepository.getContract(contractId, schoolId);
		ContractDetail contractDetail = contractRepository.getContractDetail(contractId, schoolId);
		if (contract == null) {
			logger.error("获取不到合同信息");
			return null;
		}
		if (contractDetail == null) {
			logger.error("合同信息缺失，请联系管理员");
			return null;
		}

		ContractWrapper contractWrapper = ContractWrapper.of(contract);
		contractWrapper.setContractDetail(contractDetail);
		contractWrapper.setTotalDiscount(contract.getTotalPrice() + contract.getMaterialFee() - contract.getFinalFee());

		// 封装学员信息
		List<StudentSchoolContract> sscs = studentSchoolContractRepository.getSSCByContractId(schoolId, contractId);
		if (ArrayMapTools.isEmpty(sscs)) {
			logger.error("合同学生信息缺失，请联系管理员");
			return null;
		}
		List<ContractStudentWrapper> students = Lists.newArrayList();
		for (StudentSchoolContract ssc : sscs) {
			StudentInfo si = studentInfoRepository.getStudent(ssc.getSuid());
			if (si == null) {
				logger.warn("合同学员信息不存在:" + ssc.getSuid());
				return null;
			}
			students.add(ContractStudentWrapper.ofStudentInfo(si));
		}
		contractWrapper.setStudents(students);

		// 家长信息
		contractWrapper.setParent(parentInfoService.getParentWrapper(contract.getPuid()));
		// 课包信息
		contractWrapper.setCoursePack(coursePackService.getCoursePackWrapper(school.getBrandId(), school.getProv(),
				school.getCity(), schoolId, contract.getCoursePackId()));

		List<Integer> courseIds = coursePackService.getCoursesByPackId(school.getBrandId(), school.getProv(),
				school.getCity(), schoolId, contract.getCoursePackId());
		if (ArrayMapTools.isNotEmpty(courseIds)) {
			Map<Integer, CourseWrapper> courseMap = courseService.mutiGetCourseInfoMap(school.getBrandId(), schoolId,
					courseIds, false);
			if (ArrayMapTools.isNotEmpty(courseIds)) {
				contractWrapper.setCourses(courseMap.values());
			}
		}

		Set<String> fuids = Sets.newHashSet();
		if (StringUtils.isNotEmpty(contract.getAdviser())) {
			fuids.add(contract.getAdviser());
		}
		if (StringUtils.isNotEmpty(contract.getAdviser2())) {
			fuids.add(contract.getAdviser2());
		}
		if (StringUtils.isNotEmpty(contract.getTeacher())) {
			fuids.add(contract.getTeacher());
		}
		if (StringUtils.isNotEmpty(contract.getTeacher2())) {
			fuids.add(contract.getTeacher2());
		}
		if (StringUtils.isNotEmpty(contract.getMarket())) {
			fuids.add(contract.getMarket());
		}
		// 加盟商用户信息，顾问老师市场等
		Map<String, UserSimpleWrapper> userMap = null;
		if (ArrayMapTools.isNotEmpty(fuids)) {
			userMap = fUserInfoService.mutiGetUserWrappers(fuids);
		}
		if (StringUtils.isNotEmpty(contract.getAdviser())) {
			contractWrapper.setAdviser(userMap.get(contract.getAdviser()));
		}
		if (StringUtils.isNotEmpty(contract.getAdviser2())) {
			contractWrapper.setAdviser2(userMap.get(contract.getAdviser2()));
			contractWrapper.setAdviser2Rat(contract.getAdviser2Rat());
		}
		if (StringUtils.isNotEmpty(contract.getTeacher())) {
			contractWrapper.setTeacher(userMap.get(contract.getTeacher()));
		}
		if (StringUtils.isNotEmpty(contract.getTeacher2())) {
			contractWrapper.setTeacher2(userMap.get(contract.getTeacher2()));
		}
		if (StringUtils.isNotEmpty(contract.getMarket())) {
			contractWrapper.setMarket(userMap.get(contract.getMarket()));
		}

		// 增加合同延期信息
		List<ContractPostpone> postpones = contractPostponeRepository.getPayedPostpones(schoolId, contractId);
		if (ArrayMapTools.isNotEmpty(postpones)) {
			List<ContractPostponeWrapper> pps = Lists.newArrayList();
			for (ContractPostpone cp : postpones) {
				pps.add(ContractPostponeWrapper.of(cp));
			}
			contractWrapper.setPostpones(pps);
		}

		contractWrapper.setSerialNum(school.getCode() + "" + contract.getPayTime());

		if (StringUtils.isNotEmpty(contractDetail.getTermId())) {
			contractWrapper.setTermsDetails(contractTermsRepository.getDetails(schoolId, contractDetail.getTermId()));
		}
		return contractWrapper;
	}

	public List<ContractWrapper> getParentContractList(String puid) {
		List<Contract> contracts = contractRepository.getUserContractList(puid);
		List<ContractWrapper> list = Lists.newArrayList();
		if (ArrayMapTools.isEmpty(contracts)) {
			return list;
		}
		List<String> contractIds = contracts.stream().map(it -> it.getContractId()).collect(Collectors.toList());
		List<Map<String, Object>> consumeList = studentSchoolContractRepository.mutiCalContractConsumes(contractIds);
		List<String> continueContract = studentSchoolContractRepository.getIsContinueContract(contractIds);
		Map<String, Integer> consumeOfficalLessonMap = Maps.newHashMap();
		Map<String, Integer> noScheduleOfficalLessonMap = Maps.newHashMap();
		for (Map<String, Object> map : consumeList) {
			String contractId = MapUtils.getString(map, "contractId");
			consumeOfficalLessonMap.put(contractId, MapUtils.getIntValue(map, "consumeOfficalLesson"));
			noScheduleOfficalLessonMap.put(contractId, MapUtils.getIntValue(map, "noScheduleOfficalLesson"));
		}
		for (Contract c : contracts) {
			ContractWrapper wrapper = ContractWrapper.of(c);
			wrapper.setConsumeOfficalLesson(MapUtils.getIntValue(consumeOfficalLessonMap, c.getContractId()));
			wrapper.setNoScheduleOfficalLesson(MapUtils.getIntValue(noScheduleOfficalLessonMap, c.getContractId()));
			wrapper.setIsFinished(!continueContract.contains(c.getContractId()));
			list.add(wrapper);
		}
		return list;
	}

	public boolean isHaveContract(String puid) {
		return contractRepository.isHaveContract(puid);
	}

	public Contract getSimpleContract(String contractId) {
		return contractRepository.getSimpleContract(contractId);
	}
}
