package com.jz.jar.business.proxy.common;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.collections.MapUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.common.utils.collection.OrderTool;
import com.jz.common.utils.text.StringTools;
import com.jz.jar.business.card.ItemCard;
import com.jz.jar.business.request.VideoRequest;
import com.jz.jar.business.tool.VideoTool;
import com.jz.jar.business.wrapper.VideoWrapper;
import com.jz.jar.media.enums.BrandEnum;
import com.jz.jar.media.service.MediaVideoService;
import com.jz.jar.media.service.UserCollectVideoService;
import com.jz.jar.media.service.UserPlayHistoryService;
import com.jz.jooq.media.tables.pojos.MediaVideo;
import com.jz.jooq.media.tables.pojos.UserPlayHistory;

@Lazy
@Service
public class VideoProxy {

	@Autowired
	private MediaVideoService mediaVideoService;
	@Autowired
	private UserCollectVideoService userCollectVideoService;
	@Autowired
	private UserPlayHistoryService userPlayHistoryService;

	/** 获取视频信息 */
	public VideoWrapper getVideoInfo(VideoRequest request, BrandEnum brand) {
		MediaVideo video = mediaVideoService.getVideo(request.getWid(), brand);
		if (null == video)
			return null;
		VideoWrapper videoWrapper = this.getVideoWrapper(request.getUid(), request.getPid(), video);
		// 获取之前视频的播放历史
		if (request.getWithHistory()) {
			if (StringTools.isNotEmpty(request.getUid())) {
				UserPlayHistory history = userPlayHistoryService.getHistory(request.getUid(), request.getPid(), brand);
				if (null != history && history.getWid().equals(request.getWid())) {
					videoWrapper.setPlayLength(history.getPlayLength());
				}
			}
			if (null == videoWrapper.getPlayLength())
				videoWrapper.setPlayLength(0);
		}
		if (brand == BrandEnum.playabc) {
			return videoWrapper;
		}
		// 收藏数量
		videoWrapper.setCollectCnt(userCollectVideoService.countCollectForVideo(video.getWid()));
		boolean isCollect = userCollectVideoService.isCollect(request.getUid(), request.getPid(), video.getWid());
		return videoWrapper.setIsCollect(isCollect);
	}

	public List<ItemCard> getVideoCards(VideoRequest request) {
		List<VideoWrapper> videos = this.getVideoInfos(request);
		if (ArrayMapTools.isEmpty(videos))
			return null;
		return videos.stream().map(ItemCard::of).collect(Collectors.toList());
	}
	
	/** 此方法没有写入pid，需要外层自己补全 */
	public List<VideoWrapper> getVideoInfoNotPid(VideoRequest request) {
		// 获取视频信息
		List<MediaVideo> videos = mediaVideoService.findVideos(request.getWids());
		if (ArrayMapTools.isEmpty(videos))
			return null;
		// 收藏数
		Map<String, Integer> wid2Cnt = userCollectVideoService.countCollectForVideo(request.getWids());
		List<VideoWrapper> wrappers = Lists.newArrayList();
		for (MediaVideo it : videos) {
			VideoWrapper wrapper = this.getVideoWrapper(request.getUid(), null, it);
			wrapper.setCollectCnt(MapUtils.getInteger(wid2Cnt, it.getWid(), 0));
			wrappers.add(wrapper);
		}
		return OrderTool.keepOrder(request.getWids(), wrappers, VideoWrapper::getWid);
	}
	
	public List<VideoWrapper> getVideoInfos(VideoRequest request) {
		List<VideoWrapper> videos = this.getVideoInfoNotPid(request);
		if (ArrayMapTools.isEmpty(videos))
			return null;
		videos.forEach(it -> it.setPid(request.getPid()));
		return videos;
	}
	
	public List<VideoWrapper> getVideoInfoForWid2Pid(String uid, Map<String, String> wid2Pid) {
		List<VideoWrapper> videos = this.getVideoInfoNotPid(VideoRequest.of(uid, wid2Pid.keySet()));
		if (ArrayMapTools.isEmpty(videos))
			return null;
		videos.forEach(it -> it.setPid(wid2Pid.get(it.getWid())));
		return videos;
	}
	
	/** 获取视频信息 */
	private VideoWrapper getVideoWrapper(String uid, String pid, MediaVideo video) {
		VideoWrapper result = VideoWrapper.of(video).setPid(pid);
		// 获取可播放的地址
		VideoTool.getVideoUrl(video, result);
		// 获取封面地址
		VideoTool.getSnapshotUrl(uid, video, result);
		return result;
	}
}
