package com.jz.jar.business.proxy.common;

import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.PostConstruct;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.jar.business.bean.AliyunBean;
import com.jz.jar.business.card.ItemCard;
import com.jz.jar.business.wrapper.TopicWrapper;
import com.jz.jar.media.enums.BrandEnum;
import com.jz.jar.media.service.TopicService;
import com.jz.jooq.media.tables.pojos.Topic;

@Lazy
@Service
@EnableScheduling
public class TopicProxy {

	/* id对应实例 */
	private static final Map<String, Topic> id2Topic = Maps.newLinkedHashMap();
	private static final Map<BrandEnum, List<String>> brand2Cids = Maps.newHashMap();

	@Autowired
	private TopicService topicService;

	/** 加载topic入缓存 */
	@PostConstruct
	@Scheduled(cron = "0 */5 * * * ?")
	public void loadTopicCache() {
		List<Topic> topics = topicService.findOnline();
		if (ArrayMapTools.isEmpty(topics)) {
			return;
		}
		id2Topic.clear();
		brand2Cids.clear();
		for (Topic it : topics) {
			id2Topic.put(it.getCid(), it);
			BrandEnum brand = BrandEnum.valueOf(it.getBrand());
			if (!brand2Cids.containsKey(brand))
				brand2Cids.put(brand, Lists.newArrayList());
			brand2Cids.get(brand).add(it.getCid());
		}
	}

	public List<Topic> getTopics(BrandEnum brand) {
		List<String> cids = brand2Cids.get(brand);
		if (ArrayMapTools.isEmpty(cids))
			return null;
		return cids.stream().map(id2Topic::get).collect(Collectors.toList());
	}

	public List<Topic> getTopics(Collection<String> topicIds) {
		if (ArrayMapTools.isEmpty(topicIds))
			return Lists.newArrayList();
		return Sets.newHashSet(topicIds).stream().map(id2Topic::get).collect(Collectors.toList());
	}

	public List<TopicWrapper> getTopicWrappers(Collection<String> cids) {
		return cids.stream().map(id2Topic::get).filter(it -> it != null).sorted(new Comparator<Topic>() {
			@Override
			public int compare(Topic o1, Topic o2) {
				return o1.getSeq().compareTo(o2.getSeq());
			}
		}).map(TopicProxy::packageWrapper).collect(Collectors.toList());
	}

	public List<ItemCard> getTopicCards(Collection<String> cids) {
		List<TopicWrapper> wrappers = getTopicWrappers(cids);
		if (ArrayMapTools.isEmpty(wrappers))
			return null;
		return wrappers.stream().map(ItemCard::of).collect(Collectors.toList());
	}

	public List<String> getHomeCids(BrandEnum brand) {
		List<String> cids = brand2Cids.get(brand);
		if (ArrayMapTools.isEmpty(cids))
			return null;
		return cids.stream().filter(it -> id2Topic.get(it).getPhomeShow() == 1).collect(Collectors.toList());
	}

	/** 获取首页需要展示的topics */
	public List<ItemCard> getHomeTopicCard(BrandEnum brand) {
		List<String> cids = getHomeCids(brand);
		if (ArrayMapTools.isEmpty(cids))
			return null;
		return cids.stream().map(id2Topic::get).sorted(new Comparator<Topic>() {
			@Override
			public int compare(Topic o1, Topic o2) {
				return Integer.compare(o1.getSeq(), o2.getSeq());
			}
		}).map(it -> ItemCard.of(TopicProxy.packageWrapper(it))).collect(Collectors.toList());
	}

	private static TopicWrapper packageWrapper(Topic topic) {
		return TopicWrapper.of(topic).setLeftIcon(AliyunBean.getImagesUrl(topic.getLeftIcon()))
				.setDownIcon(AliyunBean.getImagesUrl(topic.getDownIcon()))
				.setClassifyIcon(AliyunBean.getImagesUrl(topic.getClassifyIcon()));
	}
}