package com.jz.jar.business.proxy.common;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.common.utils.collection.OrderTool;
import com.jz.common.utils.text.StringTools;
import com.jz.jar.business.bean.AliyunBean;
import com.jz.jar.business.bean.MediaGlobal;
import com.jz.jar.business.enums.PictureCompress;
import com.jz.jar.business.request.UserRequest;
import com.jz.jar.business.wrapper.StudentWrapper;
import com.jz.jar.business.wrapper.UserWrapper;
import com.jz.jar.franchise.service.ContractService;
import com.jz.jar.franchise.service.ParentInfoService;
import com.jz.jar.franchise.service.StudentInfoService;
import com.jz.jar.franchise.service.StudentSchoolContractService;
import com.jz.jar.franchise.service.StudentSchoolService;
import com.jz.jar.media.enums.BrandEnum;
import com.jz.jar.media.service.FansRelationService;
import com.jz.jar.media.service.UserSignService;
import com.jz.jooq.franchise.tables.pojos.ParentInfo;
import com.jz.jooq.franchise.tables.pojos.StudentInfo;
import com.jz.jooq.media.tables.pojos.Level;

/**
 * @Title UserProxy
 * @Package com.jz.jar.media.proxy
 * @author tangjunfeng
 * @date 2019年1月11日 上午11:59:14
 * @version V1.0
 */
@Lazy
@Service
public class UserProxy {

	@Autowired
	private StudentInfoService studentInfoService;
	@Autowired
	private StudentSchoolService studentSchoolService;
	@Autowired
	private StudentSchoolContractService studentSchoolContractService;
	@Autowired
	private UserSignService userSignService;
	@Autowired
	private ContractService contractService;
	@Autowired
	private LevelProxy levelProxy;
	@Autowired
	private ParentInfoService parentInfoService;
	@Autowired
	private FansRelationService fansRelationService;
	@Autowired
	private PlayTitleProxy playTitleProxy;

	/** 获取用户信息 */
	public UserWrapper getUserInfo(String uid, BrandEnum brand) {
		return getUserInfo(uid, brand, false);
	}

	/** 获取用户信息 */
	public UserWrapper getUserInfo(String uid, BrandEnum brand, boolean isMaster) {
		ParentInfo parentInfo = parentInfoService.getParentInfoForId(uid);
		if (null == parentInfo)
			return null;
		UserWrapper user = UserWrapper.of(parentInfo);
		if (isMaster)
			user.setPhone(MediaGlobal.getPhone(parentInfo.getPhone()));
		user.setAvatar(getUserAvatar(parentInfo));
		// relation cnts
		Pair<Integer, Integer> relationCnt = fansRelationService.getRelationCnt(brand, uid);
		user.setFansCnt(relationCnt.getLeft()).setFollowCnt(relationCnt.getRight());
		// set students
		user.setStudents(getStudentsInfo(parentInfo.getPuid(), brand));
		return user;
	}

	/** 只可返回：UserWrapper || List&ltUserWrapper&gt */
	@SuppressWarnings("unchecked")
	public <T> T getSimpleParentInfo(BrandEnum brand, UserRequest request) {
		if (null == request || (StringTools.isEmpty(request.getPuid()) && ArrayMapTools.isEmpty(request.getPuids())))
			return null;
		if (null == request.getBrand())
			request.setBrand(brand);
		boolean isSingle = ArrayMapTools.isEmpty(request.getPuids());
		if (isSingle)
			request.setPuids(Arrays.asList(request.getPuid()));
		List<UserWrapper> users = getSimpleParentInfo(request);
		if (isSingle)
			return (T) ArrayMapTools.getFirst(users);
		return (T) users;
	}

	/** 获取一批用户信息 */
	public List<UserWrapper> getSimpleParentInfo(UserRequest request) {
		List<ParentInfo> parentInfos = parentInfoService.getSimpleParentInfo(request.getPuids());
		if (ArrayMapTools.isEmpty(parentInfos))
			return null;
		Map<String, List<StudentWrapper>> puid2StudnetInfo = Maps.newHashMap();
		if (request.isWithChildren()) {
			List<StudentInfo> studentInfos = studentInfoService.getSimpleStudentInfo(request.getPuids());
			if (ArrayMapTools.isNotEmpty(studentInfos)) {
				studentInfos.forEach(it -> {
					if (!puid2StudnetInfo.containsKey(it.getPuid()))
						puid2StudnetInfo.put(it.getPuid(), Lists.newArrayList());
					StudentWrapper student = StudentWrapper.of(it);
					student.setAge(getShowStudentAge(it, false));
					student.setAvatar(AliyunBean.getImagesUrl(it.getAvatar(), PictureCompress._30));
					puid2StudnetInfo.get(it.getPuid()).add(student);
				});
			}
		}

		List<String> followUids = null;
		if (null != request.getUid())
			followUids = fansRelationService.screenMyFollow(request.getBrand(), request.getUid(), request.getPuids());

		List<UserWrapper> users = Lists.newArrayList();
		for (ParentInfo it : parentInfos) {
			UserWrapper user = UserWrapper.of(it);
			if (null != followUids)
				user.setIsFollow(followUids.contains(it.getPuid()));
			user.setAvatar(getUserAvatar(it));
			if (puid2StudnetInfo.containsKey(it.getPuid()))
				user.setStudents(puid2StudnetInfo.get(it.getPuid()));
			users.add(user);
		}
		return OrderTool.keepOrder(request.getPuids(), users, UserWrapper::getUid);
	}

	/** 获取一批用户信息 */
	public Map<String, UserWrapper> getSimpleParentInfoHas(BrandEnum brand, UserRequest request) {
		List<UserWrapper> users = getSimpleParentInfo(brand, request);
		if (ArrayMapTools.isEmpty(users))
			return Maps.newHashMap();
		Map<String, UserWrapper> uid2User = Maps.newHashMap();
		users.forEach(it -> uid2User.put(it.getUid(), it));
		return uid2User;
	}

	public Level getLevelForAge(String suid, BrandEnum brand) {
		if (StringTools.isEmptyAndBlank(suid))
			return null;
		String birthday = studentInfoService.getStudentBirthday(suid);
		if (StringTools.isEmptyAndBlank(birthday))
			return null;
		return levelProxy.getLevelForAge(MediaGlobal.getAge(birthday), brand);
	}

	public String getLidForBirthday(String birthday, BrandEnum brand) {
		if (StringTools.isEmptyAndBlank(birthday))
			return null;
		Level level = levelProxy.getLevelForAge(MediaGlobal.getAge(birthday), brand);
		if (null == level)
			return null;
		return level.getLid();
	}

	/** 获取用户下的所有学员 */
	public List<StudentWrapper> getStudentsInfo(String puid, BrandEnum brand) {
		List<StudentInfo> students = studentInfoService.getStudents(puid);
		if (ArrayMapTools.isEmpty(students))
			return null;
		List<String> suids = students.stream().map(it -> it.getSuid()).collect(Collectors.toList());
		// 拆分playabc展示的内容
		Map<String, Integer> suid2Cnt = null;
		Map<String, Pair<Integer, Integer>> hasLession = null;
		List<String> containSchoolSuids = Lists.newArrayList();
		if (brand == BrandEnum.playabc) {
			suid2Cnt = userSignService.getCountDays(puid, suids);
			hasLession = studentSchoolContractService.getLessionCount(suids);
			containSchoolSuids.addAll(studentSchoolService.getContainsSchool(suids));
		}
		boolean isHaveContract = contractService.isHaveContract(puid);
		List<StudentWrapper> wrappers = Lists.newArrayList();
		for (StudentInfo st : students) {
			wrappers.add(getWrapper(st, MediaGlobal.getMapValue(suid2Cnt, st.getSuid()),
					MediaGlobal.getMapValue(hasLession, st.getSuid()), containSchoolSuids.contains(st.getSuid()),
					isHaveContract, false, brand));
		}
		return wrappers;
	}

	/** 获取用户下的所有学员 */
	public StudentWrapper getStudentInfo(String suid, boolean isSimple, BrandEnum brand) {
		StudentInfo st = studentInfoService.getStudent(suid);
		if (null == st)
			return null;
		if (isSimple || brand == BrandEnum.tomato)
			return getWrapper(st, null, null, false, false, true, brand);
		Pair<Integer, Integer> lessionCount = studentSchoolContractService.getLessionCount(st.getSuid());
		int signCnt = userSignService.countDays(st.getPuid(), suid);
		return getWrapper(st, signCnt, lessionCount, studentSchoolService.isContainsSchool(suid),
				contractService.isHaveContract(st.getPuid()), false, brand);
	}

	private StudentWrapper getWrapper(StudentInfo st, Integer signCnt, Pair<Integer, Integer> lessionCount,
			boolean isContainSchool, boolean isHaveContract, boolean isSimple, BrandEnum brand) {
		StudentWrapper student = StudentWrapper.of(st).setAge(getShowStudentAge(st, true))
				.setLevel(this.getLidForBirthday(st.getBirthday(), brand)).setSignCnt(signCnt)
				.setAvatar(getStudentAvatar(st)).setIsContainSchool(isContainSchool).setIsHaveContract(isHaveContract)
				.setQrCode(getQRCode(st.getSuid()));
		if (isSimple || brand == BrandEnum.tomato)
			return student;
		if (null != student.getSignCnt())
			student.setPlayTitle(playTitleProxy.getTitle(student.getSignCnt()));
		if (null == lessionCount)
			return student.setTotalLession(0).setRemainingLession(0);
		student.setTotalLession(lessionCount.getLeft())
				.setRemainingLession(lessionCount.getLeft() - lessionCount.getRight());
		return student;
	}

	/** 获取学员的年龄 */
	public String getShowStudentAge(StudentInfo studentInfo, boolean isShowMonth) {
		if (StringTools.isEmptyAndBlank(studentInfo.getBirthday()))
			return null;
		Pair<Integer, Integer> yearOfMonth = MediaGlobal.getAgeMonth(studentInfo.getBirthday());
		StringBuffer sb = new StringBuffer().append(getAgeContent(yearOfMonth.getLeft(), "岁"));
		if (isShowMonth)
			sb.append(getAgeContent(yearOfMonth.getRight(), "个月"));
		return sb.toString();
	}

	private static String getAgeContent(int num, String content) {
		return num <= 0 ? "" : num + content;
	}

	private static String getQRCode(String suid) {
		return "childInfo?suid=" + suid;
	}

	private static String getUserAvatar(ParentInfo parent) {
		if (StringTools.isNotEmpty(parent.getAvatar())) {
			return AliyunBean.getImagesUrl(parent.getAvatar(), PictureCompress._60);
		}
		return AliyunBean.getImagesUrl(MediaGlobal.defaultParentAvatar);
	}

	private static String getStudentAvatar(StudentInfo student) {
		if (StringTools.isNotEmpty(student.getAvatar())) {
			return AliyunBean.getImagesUrl(student.getAvatar(), PictureCompress._60);
		}
		String avatar = MediaGlobal.defaultGirlAvatar;
		if ("男".equals(student.getSex()))
			avatar = MediaGlobal.defaultBoyAvatar;
		return AliyunBean.getImagesUrl(avatar);
	}
}
