package com.jz.jar.business.proxy.common;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.collections.MapUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.common.utils.collection.OrderTool;
import com.jz.common.utils.text.StringTools;
import com.jz.jar.business.bean.AliyunBean;
import com.jz.jar.business.card.ItemCard;
import com.jz.jar.business.enums.PictureCompress;
import com.jz.jar.business.request.NextPlayVideoRequest;
import com.jz.jar.business.request.PlaylistRequest;
import com.jz.jar.business.wrapper.PlaylistWrapper;
import com.jz.jar.business.wrapper.VideoWrapper;
import com.jz.jar.franchise.service.ContractService;
import com.jz.jar.media.enums.BrandEnum;
import com.jz.jar.media.enums.PlaylistType;
import com.jz.jar.media.service.OxTabPlaylistService;
import com.jz.jar.media.service.PlaylistPicsService;
import com.jz.jar.media.service.PlaylistService;
import com.jz.jar.media.service.UserCollectService;
import com.jz.jooq.media.tables.pojos.Playlist;
import com.jz.jooq.media.tables.pojos.PlaylistPics;

@Lazy
@Service
public class PlaylistProxy {

	@Autowired
	private PlaylistService playlistService;
	@Autowired
	private PlaylistPicsService playlistPicsService;
	@Autowired
	private UserCollectService userCollectService;
	@Autowired
	private PlaylistVideoProxy playlistVideoProxy;
	@Autowired
	private ContractService contractService;
	@Autowired
	private OxTabPlaylistService oxTabPlaylistService;

	private Map<String, String> getPlaylistPics(List<PlaylistPics> pics, BrandEnum brand) {
		if (ArrayMapTools.isEmpty(pics))
			return null;
		PictureCompress compress = PictureCompress._510x280;
		if (brand == BrandEnum.tomato)
			compress = PictureCompress._60;
		Map<String, String> has = Maps.newHashMap();
		for (PlaylistPics ps : pics) {
			has.put(ps.getSize().getLiteral(), AliyunBean.getImagesUrl(ps.getUrl(), compress));
		}
		return has;
	}

	public PlaylistWrapper getSinglePlaylist(PlaylistRequest req, BrandEnum brand) {
		Playlist pl = playlistService.getOnlinePlaylist(req.getPid(), brand);
		if (null == pl)
			return null;
		PlaylistWrapper wp = PlaylistWrapper.of(pl).setIsMustHaveContract(false).setIsCollect(false);
		if (StringTools.isNotEmpty(req.getUid()))
			wp.setIsCollect(userCollectService.isCollect(req.getUid(), req.getPid(), brand));
		// 收藏数
		wp.setCollectCnt(userCollectService.countCollectForPlaylist(req.getPid()));
		// 专辑图片
		wp.setPics(getPlaylistPics(playlistPicsService.getPics(req.getPid()), brand));
		// 图片专辑已经展示完毕
		if (PlaylistType.img_txt.isEqualsName(pl.getType()))
			return wp;
		
		// playabc 验证是否牛津专辑
		if (brand == BrandEnum.playabc)
			wp.setIsMustHaveContract(oxTabPlaylistService.isOxTabPlaylist(pl.getPid()));
		// 获取下一个需要播放的视频
		if (req.getWithNextVideo()) {
			if (wp.getIsMustHaveContract()
					&& (StringTools.isEmpty(req.getUid()) || !contractService.isHaveContract(req.getUid()))) {
				return wp; // 当前用户没合同
			}
			wp.setVideo(
					playlistVideoProxy.getNextPlayVideoForHistory(req.getUid(), pl.getPid(), pl.getWid(), brand, true));
		}
		return wp;
	}

	public List<PlaylistWrapper> getMultiPlaylist(PlaylistRequest req, BrandEnum brand) {
		List<Playlist> playlistArray = playlistService.getOnlinePlaylist(req.getPids(), brand);
		if (ArrayMapTools.isEmpty(playlistArray))
			return null;
		List<String> collectPlaylist = null, oxPlaylist = null;
		// 获取已经收藏的专辑
		if (null == req.getIsCollect() && StringTools.isNotEmptyAndBlank(req.getUid()))
			collectPlaylist = userCollectService.getExistCollect(req.getUid(), req.getPids(), brand);
		// 收藏数
		Map<String, Integer> pid2CollectCnt = userCollectService.countCollectForPlaylist(req.getPids());
		
		boolean isHaveContract = false;
		if (brand == BrandEnum.playabc) { // 牛津专辑
			oxPlaylist = oxTabPlaylistService.findOxTabPlaylistId(req.getPids());
			isHaveContract = StringTools.isNotEmpty(req.getUid()) && contractService.isHaveContract(req.getUid());
		}
		
		// 专辑图片
		Map<String, List<PlaylistPics>> pid2Pics = playlistPicsService.getPicsHas(req.getPids());
		// 下一个需要播放的视频
		Map<String, VideoWrapper> pid2VideoCard = null;
		if (req.getWithNextVideo()) {
			// 只取视频专辑
			List<Playlist> videoPlaylist = playlistArray.stream()
					.filter(it -> PlaylistType.video.isEqualsName(it.getType())).collect(Collectors.toList());
			if (ArrayMapTools.isNotEmpty(videoPlaylist)) {
				NextPlayVideoRequest nextRequest = NextPlayVideoRequest.of(req.getUid(), req.getSuid(), videoPlaylist);
				nextRequest.setIsSpecial(req.getIsSpecial()).setIsEverday(req.getIsEverday());
				pid2VideoCard = playlistVideoProxy.getNextPlayVideoForHistory(nextRequest);
			}
		}
		if (null == pid2VideoCard)
			pid2VideoCard = Maps.newHashMap();

		List<PlaylistWrapper> wrappers = Lists.newArrayList();
		PlaylistWrapper wp = null;
		for (Playlist pl : playlistArray) {
			wp = PlaylistWrapper.of(pl).setIsMustHaveContract(null != oxPlaylist && oxPlaylist.contains(pl.getPid()));
			wp.setPics(this.getPlaylistPics(pid2Pics.get(pl.getPid()), brand));
			wp.setCollectCnt(MapUtils.getInteger(pid2CollectCnt, pl.getPid(), 0));
			// 是否收藏
			if (null != req.getIsCollect()) {
				wp.setIsCollect(req.getIsCollect());
			} else {
				wp.setIsCollect(collectPlaylist.contains(pl.getPid()));
			}
			if (null != req.getIsRecomm()) {
				wp.setIsRecomm(req.getIsRecomm());
			}
			if (PlaylistType.video.isEqualsName(pl.getType())) {
				if (!wp.getIsMustHaveContract() || (wp.getIsMustHaveContract() && isHaveContract))
					wp.setVideo(pid2VideoCard.get(pl.getPid()));
			}
			wrappers.add(wp);
		}
		return OrderTool.keepOrder(req.getPids(), wrappers, PlaylistWrapper::getPid);
	}

	public Map<String, PlaylistWrapper> getMultiPlaylistHas(PlaylistRequest req, BrandEnum brand) {
		List<PlaylistWrapper> playlists = this.getMultiPlaylist(req, brand);
		if (ArrayMapTools.isEmpty(playlists))
			return Maps.newHashMap();
		Map<String, PlaylistWrapper> pid2Instance = Maps.newLinkedHashMap();
		playlists.forEach(it -> pid2Instance.put(it.getPid(), it));
		return pid2Instance;
	}

	public List<ItemCard> getMultiPlaylistCards(PlaylistRequest req, BrandEnum brand) {
		List<PlaylistWrapper> wrappers = this.getMultiPlaylist(req, brand);
		if (null == wrappers)
			return null;
		return wrappers.stream().map(ItemCard::of).collect(Collectors.toList());
	}

	public Map<String, ItemCard> getMultiPlaylistCardHas(PlaylistRequest req, BrandEnum brand) {
		List<ItemCard> cards = this.getMultiPlaylistCards(req, brand);
		if (ArrayMapTools.isEmpty(cards))
			return Maps.newHashMap();
		Map<String, ItemCard> pid2Card = Maps.newLinkedHashMap();
		cards.forEach(it -> pid2Card.put(it.getPlaylist().getPid(), it));
		return pid2Card;
	}
}