package com.jz.aliyun.tools;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.net.URL;
import java.util.Calendar;
import java.util.Collection;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.aliyun.oss.OSSClient;
import com.google.common.collect.Maps;
import com.jz.aliyun.beans.AliyunConfig;

/**
 * @Title 针对于aliyun的Oss进行相关文件操作
 * @Package com.jz.aliyun.tools
 * @author tangjunfeng
 * @date 2018年7月10日 下午5:42:16
 * @version V1.0
 */
public class OssTools {

	private static final Logger logger = LoggerFactory.getLogger(OssTools.class);

	private AliyunConfig aliyunConfig;

	private OssTools() {
	}

	private OssTools(AliyunConfig aliyunConfig) {
		super();
		this.aliyunConfig = aliyunConfig;
	}

	public static OssTools getInstance(AliyunConfig aliyunConfig) {
		return new OssTools(aliyunConfig);
	}

	/** 上传本地文件 */
	public void upload(String localFilePath, String ossFilePath) throws FileNotFoundException {
		logger.info("aliyun upload file : {} => {}", localFilePath, ossFilePath);
		upload(new FileInputStream(new File(localFilePath)), ossFilePath);
	}
	
	/** 创建OSSClient实例 */
	private OSSClient getOssClient() {
		return new OSSClient(aliyunConfig.getEndpoint(), aliyunConfig.getAccessKeyId(), aliyunConfig.getAccessKeySecret());
	}
	
	/** 上传网络文件 */
	public void upload(InputStream inputStream, String ossFilePath) throws FileNotFoundException {
		// 创建OSSClient实例
		OSSClient ossClient = this.getOssClient();
		// 上传文件流
		ossClient.putObject(aliyunConfig.getBucketName(), ossFilePath, inputStream);
		// 关闭client
		ossClient.shutdown();
	}

	/** oss上的文件复制 */
	public void copy(String sourcePath, String destinationPath) throws FileNotFoundException {
		logger.info("aliyun copy file : {} => {}", sourcePath, destinationPath);
		// 创建OSSClient实例
		OSSClient ossClient = this.getOssClient();
		ossClient.copyObject(aliyunConfig.getBucketName(), sourcePath, aliyunConfig.getBucketName(), destinationPath);
		// 关闭client
		ossClient.shutdown();
	}
	
	/** oss上的文件删除 */
	public void delete(String filePath) throws FileNotFoundException {
		logger.info("aliyun delete file : {} => {}", aliyunConfig.getBucketName(), filePath);
		// 创建OSSClient实例
		OSSClient ossClient = this.getOssClient();
		ossClient.deleteObject(aliyunConfig.getBucketName(), filePath);
		// 关闭client
		ossClient.shutdown();
	}
	
	/** oss上的文件移动（like shell : mv） */
	public void move(String sourcePath, String destinationPath) {
		logger.info("aliyun mv file : {} => {}", sourcePath, destinationPath);
		// 创建OSSClient实例
		OSSClient ossClient = this.getOssClient();
		// 先复制
		ossClient.copyObject(aliyunConfig.getSourceBucketName(), sourcePath, aliyunConfig.getTargetBucketName(), destinationPath);
		// 再删除源文件
		ossClient.deleteObject(aliyunConfig.getSourceBucketName(), sourcePath);
		// 关闭client
		ossClient.shutdown();
	}
	
	/** oss上的文件可访问地址 */
	public String getUrl(String fileUrl, int expireSeconds) {
		// 创建OSSClient实例
		OSSClient ossClient = this.getOssClient();
		// 过期时间为指定秒之后
		Calendar expiration = Calendar.getInstance();
		expiration.add(Calendar.SECOND, expireSeconds);
		URL url = ossClient.generatePresignedUrl(aliyunConfig.getBucketName(), fileUrl, expiration.getTime());
		// 关闭client
		ossClient.shutdown();
		return url.toString();
	}
	
	/** oss上的文件可访问地址 */
	public Map<String, String> getUrl(Collection<String> fileUrls, int expireSeconds) {
		// 创建OSSClient实例
		OSSClient ossClient = this.getOssClient();
		// 过期时间为指定秒之后
		Calendar expiration = Calendar.getInstance();
		expiration.add(Calendar.SECOND, expireSeconds);
		Map<String, String> source2Target = Maps.newHashMap();
		for (String tempUrl : fileUrls) {
			URL url = ossClient.generatePresignedUrl(aliyunConfig.getBucketName(), tempUrl, expiration.getTime());
			source2Target.put(tempUrl, url.toString());
		}
		// 关闭client
		ossClient.shutdown();
		return source2Target;
	}
}