package com.gymchina.tiny.register.handler;

import java.util.List;

import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.GenericBeanDefinition;
import org.springframework.boot.context.properties.bind.Binder;
import org.springframework.context.annotation.Bean;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.util.ClassUtils;

import com.gymchina.tiny.register.configuration.definition.TinyDataSourceBeanDefinition;

import javassist.CannotCompileException;
import javassist.ClassPool;
import javassist.CtClass;
import javassist.CtMethod;
import javassist.LoaderClassPath;
import javassist.Modifier;
import javassist.NotFoundException;
import javassist.bytecode.AnnotationsAttribute;
import javassist.bytecode.ClassFile;
import javassist.bytecode.ConstPool;
import javassist.bytecode.ParameterAnnotationsAttribute;
import javassist.bytecode.annotation.Annotation;
import javassist.bytecode.annotation.ArrayMemberValue;
import javassist.bytecode.annotation.StringMemberValue;

public class TinySimpleDataSourceBeanDefinition {

	// create DataSource bean
	private static final String ATOMIKOS = "com.atomikos.jdbc.AtomikosDataSourceBean";
	private static final String NORMAL_BUILDER = "org.springframework.boot.jdbc.DataSourceBuilder";

	// method annotations
	private static final String CONFIGURATION = "org.springframework.context.annotation.Configuration";
	private static final String TRANSACTION_MANAGEMENT = "org.springframework.transaction.annotation.EnableTransactionManagement";
	private static final String PRIMARY = "org.springframework.context.annotation.Primary";
	private static final String BEAN = "org.springframework.context.annotation.Bean";
	private static final String PROPERTIES = "org.springframework.boot.context.properties.ConfigurationProperties";
	private static final String QUALIFIER = "org.springframework.beans.factory.annotation.Qualifier";

	// DB
	private static final String DATA_SOURCE = "javax.sql.DataSource";
	private static final String DSL_CONTEXT = "org.jooq.DSLContext";
	private static final String JDBCTEMPLATE = "org.springframework.jdbc.core.JdbcTemplate";
	private static final String NAMED_JDBCTEMPLATE = "org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate";

	// dynamic bean name
	private static final String DYNAMIC_BEAN_NAME = "com.gymchina.tiny.register.TinyAutoDbConnectionConfiguration";

	public TinySimpleDataSourceBeanDefinition(Binder binder) {
	}

	public void registerDataSources(BeanDefinitionRegistry registry, List<TinyDataSourceBeanDefinition> dataSources) {
		if (null == dataSources || dataSources.isEmpty())
			return;

		ClassPool classPool = new ClassPool();
		// get spring class loader
		classPool.insertClassPath(new LoaderClassPath(ClassUtils.getDefaultClassLoader()));
		// make dynamic connection bean
		CtClass configurationClass = classPool.makeClass(DYNAMIC_BEAN_NAME);
		ClassFile configurationClassFile = configurationClass.getClassFile();
		ConstPool configurationConstPool = configurationClassFile.getConstPool();

		// add @Configuration Annotation to class
		AnnotationsAttribute attribute = new AnnotationsAttribute(configurationConstPool,
				AnnotationsAttribute.visibleTag);
		Annotation configuration = new Annotation(CONFIGURATION, configurationConstPool);
		attribute.addAnnotation(configuration);

		// auto using atomikos for multiple data sources
		boolean isUsingAtomikos = dataSources.size() > 1;

		// add @EnableTransactionManagement Annotation to class (open transaction)
		if (isUsingAtomikos) {
			Annotation transactionManagement = new Annotation(TRANSACTION_MANAGEMENT, configurationConstPool);
			attribute.addAnnotation(transactionManagement);
			configurationClassFile.addAttribute(attribute);
		}
		
		try {
			// make (DataSource | JooQ | JdbcTemplate | NamedJdbcTemplate) bean method
			for (TinyDataSourceBeanDefinition tempBeanDefinition : dataSources) {
				makeDataSourceBeanMethod(classPool, configurationConstPool, configurationClass, tempBeanDefinition,
						isUsingAtomikos);
			}
			System.out.println("class : " + configurationClass.toString());
		} catch (NotFoundException | CannotCompileException e) {
			e.printStackTrace();
		}

		// register to spring IOC
		GenericBeanDefinition genericBeanDefinition = new GenericBeanDefinition();
		try {
			genericBeanDefinition.setBeanClass(configurationClass.toClass());
		} catch (CannotCompileException e) {
			e.printStackTrace();
		}
		registry.registerBeanDefinition("tinyAutoDbConnectionConfiguration", genericBeanDefinition);
	}

	private void makeDataSourceBeanMethod(ClassPool classPool, ConstPool constPool, CtClass configurationClass,
			TinyDataSourceBeanDefinition beanDefinition, boolean isUsingAtomikos)
			throws NotFoundException, CannotCompileException {
		String dataSourceMethodName = beanDefinition.getBeanNamePrefix() + "DataSource";
		// method body
		StringBuffer methodBody = new StringBuffer();
		methodBody.append("{\n");
		if (isUsingAtomikos) {
			methodBody.append("return new " + ATOMIKOS + "();\n");
		} else {
			methodBody.append("return " + NORMAL_BUILDER + ".create().build();\n");
		}
		methodBody.append("}");
		// make dataSource method bean
		makeJdbcMethod(classPool, constPool, configurationClass, beanDefinition, DATA_SOURCE, null,
				methodBody.toString(), "DataSource", dataSourceMethodName, true);
		
		// method arguments
		CtClass[] arguments = makeDataSourceArgument(classPool, dataSourceMethodName);

		methodBody.delete(0, methodBody.length());
		methodBody.append("{\n return org.springframework.jdbc.datasource.DataSourceTransactionManager.DataSourceTransactionManager($1); \n }");
		makeJdbcMethod(classPool, constPool, configurationClass, beanDefinition,
				"org.springframework.transaction.PlatformTransactionManager", arguments, methodBody.toString(),
				"TxManager", dataSourceMethodName, false);
		
		// make jooq
		if (beanDefinition.getJooqCtx()) {
			methodBody.delete(0, methodBody.length());
			methodBody.append("{\n return org.jooq.impl.DSL.using($1, org.jooq.SQLDialect.MYSQL); \n }");
			makeJdbcMethod(classPool, constPool, configurationClass, beanDefinition, DSL_CONTEXT, arguments,
					methodBody.toString(), "Ctx", dataSourceMethodName, false);
		}

		// make jdbcTemplate
		if (beanDefinition.getJdbctemplate()) {
			methodBody.delete(0, methodBody.length());
			methodBody.append("{\n return new " + JDBCTEMPLATE + "($1);\n }");
			makeJdbcMethod(classPool, constPool, configurationClass, beanDefinition, JDBCTEMPLATE, arguments,
					methodBody.toString(), "JdbcTemplate", dataSourceMethodName, false);
		}

		// make namedJdbcTemplate
		if (beanDefinition.getNamedJdbctemplate()) {
			methodBody.delete(0, methodBody.length());
			methodBody.append("{\n return new " + NAMED_JDBCTEMPLATE + "($1);\n }");
			makeJdbcMethod(classPool, constPool, configurationClass, beanDefinition, NAMED_JDBCTEMPLATE, arguments,
					methodBody.toString(), "NamedParameterJdbcTemplate", dataSourceMethodName, false);
		}
	}

	// make DataSource or JdbcTemplate or JooQ
	private void makeJdbcMethod(ClassPool classPool, ConstPool constPool, CtClass beanClass,
			TinyDataSourceBeanDefinition beanDefinition, String returnClass, CtClass[] arguments, String methodBody,
			String nameSuffix, String dataSourceName, boolean isConfigurationProperties)
			throws NotFoundException, CannotCompileException {
		String methodName = beanDefinition.getBeanNamePrefix() + nameSuffix;
		// method return type
		CtClass returnCtClass = classPool.get(returnClass);
		// create method definition
		CtMethod dataSourceMethod = new CtMethod(returnCtClass, methodName, arguments, beanClass);
		dataSourceMethod.setModifiers(Modifier.PUBLIC);

		// method body
		dataSourceMethod.setBody(methodBody);

		// method annotations
		AnnotationsAttribute attribute = new AnnotationsAttribute(constPool, AnnotationsAttribute.visibleTag);
		// add @Primary
		if (beanDefinition.getPrimary() && isConfigurationProperties) {
			attribute.addAnnotation(new Annotation(PRIMARY, constPool));
		}
		// add @Bean
		Annotation beanAnnotation = new Annotation(BEAN, constPool);
		// bean name property value is array
		ArrayMemberValue beanNames = new ArrayMemberValue(constPool);
		StringMemberValue[] memberValues = new StringMemberValue[] { new StringMemberValue(methodName, constPool) };
		beanNames.setValue(memberValues);
		beanAnnotation.addMemberValue("name", beanNames);
		attribute.addAnnotation(beanAnnotation);

		// add @ConfigurationProperties(prefix = "xx")
		if (isConfigurationProperties) {
			Annotation properties = new Annotation(PROPERTIES, constPool);
			properties.addMemberValue("prefix", new StringMemberValue(beanDefinition.getPropertiesPrefix(), constPool));
			attribute.addAnnotation(properties);
		}

		dataSourceMethod.getMethodInfo().addAttribute(attribute);
		
		// add @Qualifier
		if (!isConfigurationProperties) {
			ParameterAnnotationsAttribute parameterAttribute = new ParameterAnnotationsAttribute(constPool,
					ParameterAnnotationsAttribute.visibleTag);
			Annotation qualifier = new Annotation(QUALIFIER, constPool);
			qualifier.addMemberValue("value", new StringMemberValue(dataSourceName, constPool));
			Annotation[][] paramArrays = new Annotation[1][1];
			paramArrays[0][0] = qualifier;
			parameterAttribute.setAnnotations(paramArrays);
			dataSourceMethod.getMethodInfo().addAttribute(parameterAttribute);
		}
		
		// method write to class
		beanClass.addMethod(dataSourceMethod);
	}

	// make data source arguments
	private CtClass[] makeDataSourceArgument(ClassPool classPool, String dataSourceName) throws NotFoundException {
		return new CtClass[] { classPool.get(DATA_SOURCE) };
	}
}
