package com.gymchina.tiny.register.handler;

import java.io.IOException;
import java.util.List;

import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.GenericBeanDefinition;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.ImportBeanDefinitionRegistrar;
import org.springframework.core.type.AnnotationMetadata;
import org.springframework.util.ClassUtils;

import com.gymchina.tiny.register.configuration.TinyDataSourceBeanDefinition;
import com.gymchina.tiny.register.configuration.TinyRegistrarBeanDefinition;

import javassist.CannotCompileException;
import javassist.ClassPool;
import javassist.CtClass;
import javassist.CtMethod;
import javassist.LoaderClassPath;
import javassist.Modifier;
import javassist.NotFoundException;
import javassist.bytecode.AnnotationsAttribute;
import javassist.bytecode.ClassFile;
import javassist.bytecode.ConstPool;
import javassist.bytecode.annotation.Annotation;
import javassist.bytecode.annotation.ArrayMemberValue;
import javassist.bytecode.annotation.StringMemberValue;

/**
 * @Title TinySimpleRegisterBeanDefinition
 * @Package com.gymchina.tiny.register.handler
 * @author tangjunfeng
 * @date 2019年3月7日 下午4:15:21
 * @version V1.0
 */
public class TinySimpleRegisterBeanDefinition implements ImportBeanDefinitionRegistrar {

	// create DataSource bean
	private static final String ATOMIKOS = "com.atomikos.jdbc.AtomikosDataSourceBean";
	private static final String NORMAL_BUILDER = "org.springframework.boot.autoconfigure.jdbc.DataSourceBuilder";

	// method annotations
	private static final String CONFIGURATION = "org.springframework.context.annotation.Configuration";
	private static final String TRANSACTION_MANAGEMENT = "org.springframework.transaction.annotation.EnableTransactionManagement";
	private static final String PRIMARY = "org.springframework.context.annotation.Primary";
	private static final String BEAN = "org.springframework.context.annotation.Bean";
	private static final String PROPERTIES = "org.springframework.boot.context.properties.ConfigurationProperties";
	private static final String QUALIFIER = "org.springframework.beans.factory.annotation.Qualifier";

	// DB
	private static final String DATA_SOURCE = "javax.sql.DataSource";
	private static final String DSL_CONTEXT = "org.jooq.DSLContext";
	private static final String JDBCTEMPLATE = "org.springframework.jdbc.core.JdbcTemplate";
	private static final String NAMED_JDBCTEMPLATE = "org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate";

	// dynamic bean name
	private static final String DYNAMIC_BEAN_NAME = "com.gymchina.tiny.register.TinyAutoDbConnectionConfiguration";

	@Override
	public void registerBeanDefinitions(AnnotationMetadata importingClassMetadata, BeanDefinitionRegistry registry) {
		TinyRegistrarBeanDefinition beanDefinition = TinyRegistrarBeanDefinition.getInstance();
		List<TinyDataSourceBeanDefinition> beanDefainitions = beanDefinition.getBeanDefainitions();
		if (null == beanDefainitions || beanDefainitions.isEmpty())
			return;

		ClassPool classPool = new ClassPool();
		classPool.insertClassPath(new LoaderClassPath(ClassUtils.getDefaultClassLoader()));
		CtClass configurationClass = classPool.makeClass(DYNAMIC_BEAN_NAME);
		ClassFile configurationClassFile = configurationClass.getClassFile();
		ConstPool configurationConstPool = configurationClassFile.getConstPool();

		// add @Configuration Annotation to class
		AnnotationsAttribute attribute = new AnnotationsAttribute(configurationConstPool,
				AnnotationsAttribute.visibleTag);
		Annotation configuration = new Annotation(CONFIGURATION, configurationConstPool);
		attribute.addAnnotation(configuration);
		// add @EnableTransactionManagement Annotation to class
		Annotation transactionManagement = new Annotation(TRANSACTION_MANAGEMENT, configurationConstPool);
		attribute.addAnnotation(transactionManagement);
		configurationClassFile.addAttribute(attribute);

		try {
			for (TinyDataSourceBeanDefinition tempBeanDefinition : beanDefainitions) {
				makeDataSourceBeanMethod(classPool, configurationConstPool, configurationClass, tempBeanDefinition);
			}
		} catch (NotFoundException | CannotCompileException e) {
			e.printStackTrace();
		}

//		try {
//			configurationClass.writeFile("/Users/tangjunfeng/local/idea/test");
//			System.exit(0);
//		} catch (CannotCompileException | IOException e1) {
//			e1.printStackTrace();
//		}

		GenericBeanDefinition genericBeanDefinition = new GenericBeanDefinition();
		try {
			genericBeanDefinition.setBeanClass(configurationClass.toClass());
		} catch (CannotCompileException e) {
			e.printStackTrace();
		}
		registry.registerBeanDefinition("tinyAutoDbConnectionConfiguration", genericBeanDefinition);
	}

	private void makeDataSourceBeanMethod(ClassPool classPool, ConstPool constPool, CtClass configurationClass,
			TinyDataSourceBeanDefinition beanDefinition) throws NotFoundException, CannotCompileException {
		String dataSourceMethodName = beanDefinition.getBeanNamePrefix() + "DataSource";
		// method body
		StringBuffer methodBody = new StringBuffer();
		methodBody.append("{\n");
		if (beanDefinition.getIsAtomikos()) {
			methodBody.append("return new " + ATOMIKOS + "();\n");
		} else {
			methodBody.append("return " + NORMAL_BUILDER + ".create().build();\n");
		}
		methodBody.append("}");
		makeJdbcMethod(classPool, constPool, configurationClass, beanDefinition, DATA_SOURCE, null,
				methodBody.toString(), "DataSource", dataSourceMethodName, true);

		// method arguments
		CtClass[] arguments = makeDataSourceArgument(classPool, dataSourceMethodName);

		// make jooq
		if (beanDefinition.getJooqCtx()) {
			methodBody.delete(0, methodBody.length());
			methodBody.append("{\n return org.jooq.impl.DSL.using($1, org.jooq.SQLDialect.MYSQL); \n }");
			makeJdbcMethod(classPool, constPool, configurationClass, beanDefinition, DSL_CONTEXT, arguments,
					methodBody.toString(), "Ctx", dataSourceMethodName, false);
		}

		// make jdbcTemplate
		if (beanDefinition.getJdbctemplate()) {
			methodBody.delete(0, methodBody.length());
			methodBody.append("{\n return new " + JDBCTEMPLATE + "($1);\n }");
			makeJdbcMethod(classPool, constPool, configurationClass, beanDefinition, JDBCTEMPLATE, arguments,
					methodBody.toString(), "JdbcTemplate", dataSourceMethodName, false);
		}

		// make namedJdbcTemplate
		if (beanDefinition.getNamedJdbctemplate()) {
			methodBody.delete(0, methodBody.length());
			methodBody.append("{\n return new " + NAMED_JDBCTEMPLATE + "($1);\n }");
			makeJdbcMethod(classPool, constPool, configurationClass, beanDefinition, NAMED_JDBCTEMPLATE, arguments,
					methodBody.toString(), "NamedJdbcTemplate", dataSourceMethodName, false);
		}
	}

	// make DataSource or JdbcTemplate or JooQ
	private void makeJdbcMethod(ClassPool classPool, ConstPool constPool, CtClass beanClass,
			TinyDataSourceBeanDefinition beanDefinition, String returnClass, CtClass[] arguments, String methodBody,
			String nameSuffix, String dataSourceName, boolean isConfigurationProperties)
			throws NotFoundException, CannotCompileException {
		String methodName = beanDefinition.getBeanNamePrefix() + nameSuffix;
		// method return type
		CtClass returnCtClass = classPool.get(returnClass);
		// create method definition
		CtMethod dataSourceMethod = new CtMethod(returnCtClass, methodName, arguments, beanClass);
		dataSourceMethod.setModifiers(Modifier.PUBLIC);

		// method body
		dataSourceMethod.setBody(methodBody);

		// method annotations
		AnnotationsAttribute attribute = new AnnotationsAttribute(constPool, AnnotationsAttribute.visibleTag);
		// add @Primary
		if (beanDefinition.getPrimary()) {
			attribute.addAnnotation(new Annotation(PRIMARY, constPool));
		}
		// add @Bean
		Annotation beanAnnotation = new Annotation(BEAN, constPool);
		ArrayMemberValue beanNames = new ArrayMemberValue(new StringMemberValue(methodName, constPool), constPool);
//		beanAnnotation.addMemberValue("name", new StringMemberValue(methodName, constPool));
		beanAnnotation.addMemberValue("name", beanNames);
		attribute.addAnnotation(beanAnnotation);

		// add @ConfigurationProperties(prefix = "xx")
		if (isConfigurationProperties) {
			Annotation properties = new Annotation(PROPERTIES, constPool);
			properties.addMemberValue("prefix", new StringMemberValue(beanDefinition.getPropertiesPrefix(), constPool));
			attribute.addAnnotation(properties);
		} else { // add @@Qualifier
			Annotation qualifier = new Annotation(QUALIFIER, constPool);
			qualifier.addMemberValue("value", new StringMemberValue(dataSourceName, constPool));
			attribute.addAnnotation(qualifier);
		}

		dataSourceMethod.getMethodInfo().addAttribute(attribute);

		// method write to class
		beanClass.addMethod(dataSourceMethod);
	}

	// make data source arguments
	private CtClass[] makeDataSourceArgument(ClassPool classPool, String dataSourceName) throws NotFoundException {
		// argument definition
		CtClass dataSourceClass = classPool.get(DATA_SOURCE);
		ClassFile dataSourceClassFile = dataSourceClass.getClassFile();
		ConstPool dataSourceConstPool = dataSourceClassFile.getConstPool();

		// argument annotation
		AnnotationsAttribute attribute = new AnnotationsAttribute(dataSourceConstPool, AnnotationsAttribute.visibleTag);
		// add @@Qualifier
		Annotation qualifier = new Annotation(QUALIFIER, dataSourceConstPool);
		qualifier.addMemberValue("value", new StringMemberValue(dataSourceName, dataSourceConstPool));
		attribute.addAnnotation(qualifier);
		dataSourceClassFile.addAttribute(attribute);
		return new CtClass[] { dataSourceClass };
	}
}
